//////////////////////////////////////////////////////////////////////////////
// Skel
// Header file
//
// A Model Real-Time Interactive C++/OpenGL/GLUT Animator
// Adapted by Michael Fisher from
//     Skel in C by George Francis, Stuart Levy, Glenn Chappell, Chris Hartman
//     email: gfrancis@math.uiuc.edu
//
// (C) 1994-2004 Board of Trustees University of Illinois
//
//////////////////////////////////////////////////////////////////////////////

#include "LinkedList.h"

// Global defines
#ifndef NULL
#define NULL 0L
#endif
#ifndef PI
#define PI 3.1415926535897932384626433832795f
#endif

// Forward declarations of global event handling functions
void DisplayFunc();
void KeyboardFunc(unsigned char key, int x, int y);
void SpecialFunc(int key, int x, int y);
void MouseFunc(int button, int state, int x, int y);
void MotionFunc(int x, int y);
void ReshapeFunc(int width, int height);
void IdleFunc();

// Forward declarations of important classes
class SkelWindow;
class Light;
class Object;
class CommandArgument;
class CommandValue;
class FrameRateCounter;

// Globals
// The one and only SkelWindow object
SkelWindow* window;


//////////////////////////////////////////////////////////////////////////////
//
// FrameRateCounter
//
// Calculates and limits the application frame rate
//
//////////////////////////////////////////////////////////////////////////////
class FrameRateCounter
{
private:
	long History[9];
	int Index;

public:
	// Default constructor
	FrameRateCounter();
	
	// Schedule the time for the next frame
	void ScheduleNextFrame();
	// Returns true if it is time to draw another frame
	bool ReadyForNextFrame();
	// Calculate the frame rate
	float GetFrameRate();

private:
	// Get the current time in miliseconds (since some base time)
	long GetTime();
};


//////////////////////////////////////////////////////////////////////////////
//
// SkelWindow
//
// This object encapsulates the GLUT window where all user interaction occurs
//
//////////////////////////////////////////////////////////////////////////////
class SkelWindow
{
private:
	enum InteractionMode { TurnMode, FlyMode };

	// Window creation parameters:
	int window_x;
	int window_y;
	int window_width;
	int window_height;
	bool window_fullscreen;

	// Camera motion
	float CameraMatrix[16];
	InteractionMode Mode;
	float RotationAmount;
	float TranslationAmount;
	float ScaleFactor;
	float FocalFactor;
	float RearClippingPlane;
	float NoseDistance;

	// Stars background
	static const int NumberOfStars = 8192;
	float StarsPositions[NumberOfStars][3];
	float StarsMatrix[16];

	LinkedList<Object*> SceneObjects;
	LinkedList<Light*> SceneLights;

	// Mouse and keyboard state
	int mouse_x, mouse_y;		// mouse position
	bool left_button, middle_button, right_button;		// flag indicates if the button is currently pressed
	bool shift_key, ctrl_key;		// flag indicates if the key is currently pressed
	bool key_press[128];		// flag for presses to each key
	bool special_key_press[128];		// flag for presses to each special key

	// Drawing options state
	bool MovementEnabled;
	bool DepthBufferEnabled;
	bool AnimationEnabled;
	bool BinocularEnabled;
	int OverlayLevel;

	// Light applied evenly to all objects
	float GlobalAmbientLight[4];

	// Track the times that frames are drawn
	FrameRateCounter Counter;

	// Current position in printing to the overlay (HUD)
	float CurrentOverlayPosition;

	// Flag set to true when it's time to exit
	bool ExitApplication;

public:
	// Command-line arguments used for initializing
	LinkedList<CommandArgument*> Arguments;

public:
	// Default constructor
	SkelWindow();

	// Create the window and initialize GLUT
	void Initialize(int argc, char** argv);
	// Show the window and start rendering the scene
	void Run();
	// Returns true if the specified key is pressed
	bool IsKeyPressed(unsigned char key);
	// Returns true if the specified special key is pressed
	bool IsSpecialKeyPressed(int key);
	// Print the specified text to the overlay (HUD)
	void PrintOverlayText(char* text);

	// Draw all objects
	void Display();
	// Handle mouse movement
	void Motion(int x, int y);
	// Handle mouse clicks
	void Mouse(int button, int state, int x, int y);
	// Perform operations during idle time
	void Idle();
	// Handle key presses
	void Keyboard(unsigned char key, int x, int y);
	// Handle special key presses
	void Special(int key, int x, int y);
	// Handle the window resize
	void Reshape(int width, int height);

protected:
	// Create the scene objects
	void SetupSceneObjects();
	// Change the camera motion based on recent key presses
	void HandleKeyPresses();
	// Move the camera
	void MoveCamera();
	// Load the default values for everything
	void Reset();
	// Create the lights
	void SetupLighting();

	// Destructor
	~SkelWindow();

private:
	// Read settings from the command-line arguments
	void ReadArguments(int argc, char** argv);
	// Pause the program for a short time
	void Delay();
	// Draw the stars background
	void DisplayStars();
	// Draw the text overlay
	void DisplayOverlay();
	// Print a string at the specified coordinates on the 2D surface
	void DisplayString(float x, float y, char* text);
};


//////////////////////////////////////////////////////////////////////////////
//
// Light
//
// Sets up an OpenGL light source
//
//////////////////////////////////////////////////////////////////////////////
class Light
{
private:
	// The coordinates of this light
	float Position[4];
	// The direction the light faces
	float Direction[3];
	// The ambient color and intensity produced
	float AmbientColor[4];
	// The diffuse color and intensity produced
	float DiffuseColor[4];
	// The specular color and intensity produced
	float SpecularColor[4];
	// Attenuation of light intensity based on the angle between a surface and light direction
	float SpotExponent;
	// The maximum angle reached by the light
	float SpotCutoff;
	// The rate at which the light intensity decreases with distance
	float LinearAttenuation;

	// The OpenGL light number
	int LightNumber;

	// The next available OpenGL light number
	static int NextLightNumber;

public:
	// Default constructor
	Light();

	// Enable this light
	void Enable();
	// Disable this light
	void Disable();

	// Setters
	void SetPosition(float x, float y, float z, float alpha);
	void SetDirection(float x, float y, float z);
	void SetAmbientColor(float red, float green, float blue, float alpha);
	void SetDiffuseColor(float red, float green, float blue, float alpha);
	void SetSpecularColor(float red, float green, float blue, float alpha);
	void SetSpotExponent(float exponent);
	void SetSpotCutoff(float cutoff);
	void SetLinearAttenuation(float attenuation);
};


//////////////////////////////////////////////////////////////////////////////
//
// CommandArgument
//
// Holds a command-line argument passed to the application
//
//////////////////////////////////////////////////////////////////////////////
class CommandArgument
{
private:
	char* Name;
	LinkedList<CommandValue> Values;

public:
	// Constructor
	CommandArgument(char* name);

	// Adds a value for this argument
	void InsertValue(char* value);
	// Return the name of this argument
	char* GetName();
	// Return the values that followed this argument on the command-line
	LinkedList<CommandValue>* GetValues();
};

//////////////////////////////////////////////////////////////////////////////
//
// CommandValue
//
// Holds a command-line argument values passed to the application
//
//////////////////////////////////////////////////////////////////////////////
class CommandValue
{
private:
	char* Value;

public:
	// Default constructor
	CommandValue();
	// Normal constructor
	CommandValue(char* value);

	// Return the value as a string
	char* GetString();
	// Return the value as an int
	int GetInt();
	// Return the value as a float
	float GetFloat();
};


//////////////////////////////////////////////////////////////////////////////
//
// Object
//
// This is the base class for all objects in the scene
//
//////////////////////////////////////////////////////////////////////////////
class Object
{
private:
	LinkedList<Object*> children;

protected:
	int AnimationStep;
	// Defines how to move this object relative to its parent
	float Position[3];
	// Defines how to scale this object
	float Size[3];
	// Defines how to rotate this object around each axis
	float Rotation[3];

public:
	// Default constructor
	Object();

	// Destructor
	virtual ~Object();

	// Add a child object
	void InsertChild(Object* object);

	// Restore the default settings for this object and all its children
	void ResetAll();
	// Draw this object and all its children
	void DrawAll();
	// Draw2D for this object and all its children
	void Draw2DAll();
	// Step forward the animation for this object and all its children
	void StepAnimationAll();
	// Set the object position
	void SetPosition(float x, float y, float z);
	// Set the object size
	void SetSize(float x, float y, float z);
	// Set the object rotation for each axis
	void SetRotation(float angle_x, float angle_y, float angle_z);

protected:
	// Restore the object's default settings
	virtual void Reset() = 0;
	// Draw this object
	virtual void Draw() = 0;
	// Draw 2D items such as messages on the overlay (HUD)
	virtual void Draw2D() = 0;
	// Called to inform the object that animation should advance forward by one step
	virtual void StepAnimation() = 0;
};

//////////////////////////////////////////////////////////////////////////////
//
// Torus
//
//////////////////////////////////////////////////////////////////////////////
class Torus : public Object
{
protected:
	float Gap;
	float DrawFactor;
	bool NormalVectors;

	static const unsigned char IncreaseGapKey = 'G';
	static const unsigned char DecreaseGapKey = 'g';
	static const unsigned char ToggleNormalVectorsKey = 't';

public:
	// Default constructor
	Torus();

protected:
	// Restore the default settings for the torus
	void Reset();
	// Draw the torus
	void Draw();
	// Draw 2D items such as messages on the overlay (HUD)
	void Draw2D();
	// Step forward the animation for the torus
	void StepAnimation();
};


//////////////////////////////////////////////////////////////////////////////
//
// Cube
//
//////////////////////////////////////////////////////////////////////////////
class Cube : public Object
{
public:
	// Default constructor
	Cube();

protected:
	// Restore the default settings for the cube
	void Reset();
	// Draw the cube
	void Draw();
	// Draw 2D items such as messages on the overlay (HUD)
	void Draw2D();
	// Step forward the animation for the cube
	void StepAnimation();
};


//////////////////////////////////////////////////////////////////////////////
//
// Tetrahedron
//
//////////////////////////////////////////////////////////////////////////////
class Tetrahedron : public Object
{
public:
	// Default constructor
	Tetrahedron();

protected:
	// Restore the default settings for the tetrahedron
	void Reset();
	// Draw the tetrahedron
	void Draw();
	// Draw 2D items such as messages on the overlay (HUD)
	void Draw2D();
	// Step forward the animation for the tetrahedron
	void StepAnimation();
};



