//////////////////////////////////////////////////////////////////////////////
// Skel
// Source file
//
// A Model Real-Time Interactive C++/OpenGL/GLUT Animator
// Adapted by Michael Fisher from
//     Skel in C by George Francis, Stuart Levy, Glenn Chappell, Chris Hartman
//     email: gfrancis@math.uiuc.edu
//
// (C) 1994-2004 Board of Trustees University of Illinois
//
//////////////////////////////////////////////////////////////////////////////

#include "skel.h"
#include <stdlib.h>
#include <stdio.h>
//#include <GL/glut.h>
#include <glut.h>
#include <math.h>

// The following includes are for thread sleeping
#ifdef WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

// The following includes are for framerate time calculations
#ifdef WIN32
#include <time.h>
#else
#include <sys/time.h>
#endif

// Enable or disable the frame rate limiter by setting this to 1 or 0
#define ENABLE_FRAME_RATE_LIMITER 1


//////////////////////////////////////////////////////////////////////////////
//
// Global functions
//
//////////////////////////////////////////////////////////////////////////////

// Program entry point
int main(int argc, char** argv)
{
	// Create the window
	window = new SkelWindow();
	window->Initialize(argc, argv);

	// Show the window and begin displaying
	window->Run();

	return 0;
}

// GLUT calls this function when the scene needs to be rendered
void DisplayFunc()
{
	// Render the scene
	window->Display();
}

// GLUT calls this function when a key is pressed
void KeyboardFunc(unsigned char key, int x, int y)
{
	window->Keyboard(key, x, y);
}

// GLUT calls this function when a special key is pressed
void SpecialFunc(int key, int x, int y)
{
	window->Special(key, x, y);
}

// GLUT calls this function when a mouse button is pressed
void MouseFunc(int button, int state, int x, int y)
{
	window->Mouse(button, state, x, y);
}

// GLUT calls this function when the mouse moves
void MotionFunc(int x, int y)
{
	window->Motion(x, y);
}

// GLUT calls this function when the window is resized
void ReshapeFunc(int width, int height)
{
	window->Reshape(width, height);
}

// GLUT calls this function when there is nothing else to do
void IdleFunc()
{
	window->Idle();
}


//////////////////////////////////////////////////////////////////////////////
//
// SkelWindow
//
// This object encapsulates the GLUT window where all user interaction occurs
//
//////////////////////////////////////////////////////////////////////////////

// Default constructor
SkelWindow::SkelWindow() : ExitApplication(false)
{
	// Set default window parameters of the window creation
	window_x = 0;
	window_y = 0;
	window_width = 640;
	window_height = 480;
	window_fullscreen = false;

	// Set initial mouse and keybord state
	left_button = false;
	middle_button = false;
	right_button = false;
	shift_key = false;
	ctrl_key = false;
	memset(key_press, 0, sizeof(key_press));
	memset(special_key_press, 0, sizeof(special_key_press));
}

// Destructor
SkelWindow::~SkelWindow()
{
	// Release all objects
	for(LinkedList<Object*>::Iterator object_itr = SceneObjects.begin(); object_itr != SceneObjects.end(); ++object_itr)
	{
		delete *object_itr;
	}

	// Release all lights
	for(LinkedList<Light*>::Iterator light_itr = SceneLights.begin(); light_itr != SceneLights.end(); ++light_itr)
	{
		delete *light_itr;
	}

	// Release command-line arguments
	for(LinkedList<CommandArgument*>::Iterator argument_itr = Arguments.begin(); argument_itr != Arguments.end(); ++argument_itr)
	{
		delete *argument_itr;
	}
}

// Create the window and initialize GLUT
void SkelWindow::Initialize(int argc, char** argv)
{
	// Initialize GLUT
	glutInit(&argc, argv);

	// Read the command-line arguments
	ReadArguments(argc, argv);

	for(LinkedList<CommandArgument*>::Iterator itr = Arguments.begin(); itr != Arguments.end(); ++itr)
	{
		CommandArgument* argument = *itr;
		LinkedList<CommandValue>* values;

		// Read the window initialization arguments
		if(strcmp(argument->GetName(), "w") == 0)
		{
			values = argument->GetValues();
			if(values->size() > 0)
			{
				int win = values->front().GetInt();
				if(win == 1)
				{
					window_x = 100;
					window_y = 100;
					window_width = 640;
					window_height = 480;
					window_fullscreen = false;
				}
				else if(win == 2)
				{
					window_x = 0;
					window_y = 0;
					window_fullscreen = true;
				}
			}
		}
	}

	// Place the stars
	for(int i = 0; i < NumberOfStars; ++i)
	{
		float magnitude = 0;
		for(int j = 0; j < 3; ++j)
		{
			float temp = (float)rand() / RAND_MAX - 0.5;
			StarsPositions[i][j] = temp;
			magnitude += temp * temp;
		}
		magnitude = sqrt(magnitude);
		for(int j = 0; j < 3; ++j)
		{
			StarsPositions[i][j] /= magnitude;
		}
	}
}

// Create the lights
void SkelWindow::SetupLighting()
{
	// Specify that we'll light the backs of polygons too
	const float yes = 1.0f;
	glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, &yes);

	Light* light;

	// create light that is a strong spotlight next to your head
	light = new Light();
	light->SetPosition(1.0f, 1.0f, 3.0f, 1.0f);
	light->SetDiffuseColor(0.0f, 0.0f, 0.0f, 1.0f);
	light->SetSpecularColor(1.0f, 1.0f, 1.0f, 1.0f);
	SceneLights.push_back(light);
	light->Enable();

	// create light that is a diffuse light near the floor
	light = new Light();
	light->SetPosition(-1.0f, -1.0f, 5.0f, 1.0f);
	light->SetDiffuseColor(1.0f, 1.0f, 1.0f, 1.0f);
	light->SetSpecularColor(0.0f, 0.0f, 0.0f, 1.0f);
	SceneLights.push_back(light);
	light->Enable();
}

// Create the scene objects
void SkelWindow::SetupSceneObjects()
{
	Torus* torus = new Torus();
	SceneObjects.push_back(torus);
	
	Cube* cube = new Cube();
	cube->SetPosition(-1.3f, 0.9f, 0.2f);
	cube->SetSize(2.0f, 0.5f, 0.5f);
	cube->SetRotation(0.0f, 90.0f, 15.0f);
	torus->InsertChild(cube);

	Tetrahedron* tetrahedron = new Tetrahedron();
	tetrahedron->SetSize(0.4f, 0.4f, 0.4f);
	SceneObjects.push_back(tetrahedron);
}

// Show the window and start rendering the scene
void SkelWindow::Run()
{
	const char* window_title = "<* illiSkel in C++/OpenGL/GLUT *>";

	// Schedule the first frame
	Counter.ScheduleNextFrame();

	// Set up the window position and size
	glutInitWindowPosition(window_x, window_y);
	glutInitWindowSize(window_width, window_height);

	// Use RGBA color, double buffer animation, and the depth
	// buffer for depth testing
	glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH);

	glutCreateWindow(window_title);
	
	if(window_fullscreen)
	{
		glutFullScreen();
	}

	glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
	glClearDepth(1.0f);
	glEnable(GL_NORMALIZE);			// scale normals to unit length
	glShadeModel(GL_SMOOTH);		// smooth (phong) shading

	// this allows the glColor function to set a material's ambient and diffuse reflectiveness
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glEnable(GL_COLOR_MATERIAL);

	// Attach our functions to GLUT events
	glutDisplayFunc(DisplayFunc);
	glutKeyboardFunc(KeyboardFunc);
	glutSpecialFunc(SpecialFunc);
	glutMouseFunc(MouseFunc);
	glutMotionFunc(MotionFunc);
	glutPassiveMotionFunc(MotionFunc);
	glutReshapeFunc(ReshapeFunc);
	glutIdleFunc(IdleFunc);

	// Create lights
	SetupLighting();

	// Create the scene objects
	SetupSceneObjects();

	// Load the default values
	Reset();

	glutMainLoop();
}

void SkelWindow::Display()
{
	LinkedList<Object*>::Iterator itr;

	// Set the time for the next frame
	Counter.ScheduleNextFrame();

	// Check for key presses that will modify the way we draw
	HandleKeyPresses();

	// Step forward the animation
	if(AnimationEnabled)
	{
		for(itr = SceneObjects.begin(); itr != SceneObjects.end(); ++itr)
		{
			(*itr)->StepAnimationAll();
		}
	}

	// Position the camera based on mouse actions
	if(MovementEnabled)
	{
		MoveCamera();
	}

	// Apply the global ambient light
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, GlobalAmbientLight);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrustum(-ScaleFactor * window_width / window_height, ScaleFactor * window_width / window_height, -ScaleFactor, ScaleFactor, ScaleFactor * FocalFactor, RearClippingPlane);

	if(BinocularEnabled)
	{
		glViewport(0.0f, window_height >> 2, window_width >> 1, window_height >> 1);
	}
	else
	{
		glViewport(0.0f, 0.0f, window_width, window_height);
	}

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	// Draw the stars background
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	DisplayStars();

	if(BinocularEnabled)
	{
		glTranslatef(-NoseDistance, 0.0f, 0.0f);
	}

	glMultMatrixf(CameraMatrix);

	// Enable or disable the depth buffer test
	if(DepthBufferEnabled)
	{
		glEnable(GL_DEPTH_TEST);
	}
	else
	{
		glDisable(GL_DEPTH_TEST);
	}

	glEnable(GL_LIGHTING);

	// Draw all the scene objects
	for(itr = SceneObjects.begin(); itr != SceneObjects.end(); ++itr)
	{
		(*itr)->DrawAll();
	}

	// Draw everything a second time is binocular mode
	if(BinocularEnabled)
	{
		glViewport(window_width >> 1, window_height >> 2, window_width >> 1, window_height >> 1);

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		// Draw the stars background
		glDisable(GL_DEPTH_TEST);
		glDisable(GL_LIGHTING);
		DisplayStars();

		glTranslatef(NoseDistance, 0.0f, 0.0f);

		glMultMatrixf(CameraMatrix);

		// Enable or disable the depth buffer test
		if(DepthBufferEnabled)
		{
			glEnable(GL_DEPTH_TEST);
		}
		else
		{
			glDisable(GL_DEPTH_TEST);
		}

		glEnable(GL_LIGHTING);

		// Draw all the scene objects
		for(itr = SceneObjects.begin(); itr != SceneObjects.end(); ++itr)
		{
			(*itr)->DrawAll();
		}
	}

	glViewport(0.0f, 0.0f, window_width, window_height);

	if(OverlayLevel != 0)
	{
		glDisable(GL_LIGHTING);
		DisplayOverlay();
		glEnable(GL_LIGHTING);
	}

	glutSwapBuffers();
}

void SkelWindow::Motion(int x, int y)
{
	// Update the current mouse position
	mouse_x = x;
	mouse_y = y;
}

void SkelWindow::Mouse(int button, int state, int x, int y)
{
	// Update the current key modifiers
	int modifiers = glutGetModifiers();
	shift_key = ((modifiers & GLUT_ACTIVE_SHIFT) != 0);
	ctrl_key = ((modifiers & GLUT_ACTIVE_CTRL) != 0);

	// Update the current mouse position
	Motion(x, y);

	// Update the current mouse button state
	switch(button)
	{
	case GLUT_LEFT_BUTTON:
		left_button = (state == GLUT_DOWN);
		break;
	case GLUT_MIDDLE_BUTTON:
		middle_button = (state == GLUT_DOWN);
		break;
	case GLUT_RIGHT_BUTTON:
		right_button = (state == GLUT_DOWN);
		break;
	}
}

void SkelWindow::Idle()
{
	if(ExitApplication == true)
	{
		exit(0);
	}

	if(Counter.ReadyForNextFrame() == true)
	{
		// Redraw the scene
		glutPostRedisplay();
	}
	else
	{
		// Wait a bit longer
		Delay();
	}
}

void SkelWindow::Keyboard(unsigned char key, int x, int y)
{
	// Update the current mouse position
	Motion(x, y);

	// Flag this key as pressed
	key_press[key & 127] = true;
}

void SkelWindow::Special(int key, int x, int y)
{
	// Update the current mouse position
	Motion(x, y);

	// Flag this key as pressed
	special_key_press[key & 127] = true;
}

void SkelWindow::Reshape(int width, int height)
{
	window_width = width;
	window_height = height;
}

// Read settings from the command-line arguments
void SkelWindow::ReadArguments(int argc, char** argv)
{
	CommandArgument* argument = NULL;

	// Read the command-line arguments
	for(int i = 0; i < argc; ++i)
	{
		if(argv[i][0] == '-')
		{
			// Start of a new argument
			argument = new CommandArgument(&argv[i][1]);
			Arguments.push_back(argument);
		}
		else
		{
			// Add a value to the current argument
			if(argument != NULL)
			{
				argument->InsertValue(argv[i]);
			}
		}
	}
}

// Change the camera motion based on recent key presses
void SkelWindow::HandleKeyPresses()
{
	const unsigned char ModeToggleKey = ' ';
	const unsigned char MovementToggleKey = '=';
	const unsigned char DepthBufferToggleKey = '-';
	const unsigned char AnimationToggleKey = 'h';
	const unsigned char BinocularToggleKey = 'v';
	const unsigned char IncreaseOverlayLevelKey = 'w';
	const unsigned char DecreaseOverlayLevelKey = 'W';
	const unsigned char IncreaseNoseDistanceKey = 'N';
	const unsigned char DecreaseNoseDistanceKey = 'n';
	const unsigned char IncreaseTranslationAmountKey = 's';
	const unsigned char DecreaseTranslationAmountKey = 'S';
	const unsigned char IncreaseRotationAmountKey = 'q';
	const unsigned char DecreaseRotationAmountKey = 'Q';
	const unsigned char IncreaseFocalFactorKey = 'o';
	const unsigned char DecreaseFocalFactorKey = 'O';
	const unsigned char IncreaseScaleFactorKey = 'I';
	const unsigned char DecreaseScaleFactorKey = 'i';
	const unsigned char IncreaseRearClippingPlaneKey = 'p';
	const unsigned char DecreaseRearClippingPlaneKey = 'P';
	const unsigned char IncreaseGlobalAmbientLightKey = 'A';
	const unsigned char DecreaseGlobalAmbientLightKey = 'a';
	const unsigned char ResetKey = 'z';
	const unsigned char ExitKey = 27;

	// Check for special keys to control how we draw
	MovementEnabled ^= IsKeyPressed(MovementToggleKey);
	DepthBufferEnabled ^= IsKeyPressed(DepthBufferToggleKey);
	AnimationEnabled ^= IsKeyPressed(AnimationToggleKey);
	BinocularEnabled ^= IsKeyPressed(BinocularToggleKey);

	// Cycle through the three levels of overlay display
	if(IsKeyPressed(IncreaseOverlayLevelKey))
	{
		OverlayLevel = (OverlayLevel == 2)?(0):(OverlayLevel + 1);
	}
	if(IsKeyPressed(DecreaseOverlayLevelKey))
	{
		OverlayLevel = (OverlayLevel == 0)?(2):(OverlayLevel - 1);
	}

	// Control binocular nose distance
	if(IsKeyPressed(IncreaseNoseDistanceKey))
	{
		NoseDistance += 0.001f;
	}
	if(IsKeyPressed(DecreaseNoseDistanceKey))
	{
		NoseDistance -= 0.001f;
	}

	// Control translation amount (speed)
	if(IsKeyPressed(IncreaseTranslationAmountKey))
	{
		TranslationAmount += 0.02f;
	}
	if(IsKeyPressed(DecreaseTranslationAmountKey))
	{
		TranslationAmount -= 0.02f;
	}

	// Control rotation amount (speed)
	if(IsKeyPressed(IncreaseRotationAmountKey))
	{
		RotationAmount += 0.01f;
	}
	if(IsKeyPressed(DecreaseRotationAmountKey))
	{
		RotationAmount -= 0.01f;
	}

	// Control telephoto focus point
	if(IsKeyPressed(IncreaseFocalFactorKey))
	{
		FocalFactor *= 1.1f;
	}
	if(IsKeyPressed(DecreaseFocalFactorKey))
	{
		FocalFactor /= 1.1f;
	}

	// Control world scale factor
	if(IsKeyPressed(IncreaseScaleFactorKey))
	{
		ScaleFactor *= 1.1f;
	}
	if(IsKeyPressed(DecreaseScaleFactorKey))
	{
		ScaleFactor /= 1.1f;
	}

	// Control the distance of the rear clipping plane
	if(IsKeyPressed(IncreaseRearClippingPlaneKey))
	{
		RearClippingPlane *= 1.01f;
	}
	if(IsKeyPressed(DecreaseRearClippingPlaneKey))
	{
		RearClippingPlane /= 1.01f;
	}

	// Control the amount of light applied evenly to all objects
	if(IsKeyPressed(IncreaseGlobalAmbientLightKey))
	{
		GlobalAmbientLight[0] += 0.1f;
		GlobalAmbientLight[1] += 0.1f;
		GlobalAmbientLight[2] += 0.1f;
	}
	if(IsKeyPressed(DecreaseGlobalAmbientLightKey))
	{
		GlobalAmbientLight[0] -= 0.1f;
		GlobalAmbientLight[1] -= 0.1f;
		GlobalAmbientLight[2] -= 0.1f;
	}

	// Set the current interaction mode
	if(IsKeyPressed(ModeToggleKey))
	{
		switch(Mode)
		{
		case TurnMode:
			Mode = FlyMode;
			break;
		case FlyMode:
			Mode = TurnMode;
			break;
		}
	}

	// Restore all default settings
	if(IsKeyPressed(ResetKey))
	{
		Reset();
	}

	// Check if we should exit
	if(IsKeyPressed(ExitKey))
	{
		ExitApplication = true;
	}
}

void SkelWindow::MoveCamera()
{

	const float LargeRotation = 10.0f;
	const float SmallRotation = 1.0f;

	// Determine how far the mouse pointer is from the window center
	int dx, dy;
	dx = mouse_x - (window_width >> 1);
	if(dx < 5 && dx > -5)
		dx = 0;		// don't move any if the mouse is close to the center
	dy = mouse_y - (window_height >> 1);
	if(dy < 5 && dy > -5)
		dy = 0;		// don't move any if the mouse is close to the center

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	if(Mode == TurnMode)
	{
		glTranslatef(CameraMatrix[12], CameraMatrix[13], CameraMatrix[14]);
	}

	glRotatef(dx * RotationAmount, 0.0f, 1.0f, 0.0f);
	glRotatef(dy * RotationAmount, 1.0f, 0.0f, 0.0f);

	if(left_button == true && right_button == false)
	{
		if(shift_key == true)
		{
			glRotatef(LargeRotation, 0.0f, 0.0f, 1.0f);
		}
		else
		{
			glRotatef(SmallRotation, 0.0f, 0.0f, 1.0f);
		}
	}
	else if(left_button == false && right_button == true)
	{
		if(shift_key == true)
		{
			glRotatef(-LargeRotation, 0.0f, 0.0f, 1.0f);
		}
		else
		{
			glRotatef(-SmallRotation, 0.0f, 0.0f, 1.0f);
		}
	}
	
	if(Mode == FlyMode)
	{
		glPushMatrix();
		glMultMatrixf(StarsMatrix);
		glGetFloatv(GL_MODELVIEW_MATRIX, StarsMatrix);
		glPopMatrix();
	}

	if(middle_button == true)
	{
		if(shift_key)
		{
			glTranslatef(0.0f, 0.0f, -TranslationAmount);
		}
		else
		{
			glTranslatef(0.0f, 0.0f, TranslationAmount);
		}
	}

	if(IsSpecialKeyPressed(GLUT_KEY_UP))
	{
		glTranslatef(0.0f, 0.0f, TranslationAmount);
	}
	if(IsSpecialKeyPressed(GLUT_KEY_DOWN))
	{
		glTranslatef(0.0f, 0.0f, -TranslationAmount);
	}
	if(IsSpecialKeyPressed(GLUT_KEY_LEFT))
	{
		glTranslatef(-TranslationAmount, 0.0f, 0.0f);
	}
	if(IsSpecialKeyPressed(GLUT_KEY_RIGHT))
	{
		glTranslatef(TranslationAmount, 0.0f, 0.0f);
	}
	if(IsSpecialKeyPressed(GLUT_KEY_PAGE_UP))
	{
		glTranslatef(0.0f, TranslationAmount, 0.0f);
	}
	if(IsSpecialKeyPressed(GLUT_KEY_PAGE_DOWN))
	{
		glTranslatef(0.0f, -TranslationAmount, 0.0f);
	}

	if(Mode == TurnMode)
	{
		glTranslatef(-CameraMatrix[12], -CameraMatrix[13], -CameraMatrix[14]);
	}

	glMultMatrixf(CameraMatrix);
	glGetFloatv(GL_MODELVIEW_MATRIX, CameraMatrix);
	glPopMatrix();
}

// Load the default values for everything
void SkelWindow::Reset()
{
	const float InitialCameraMatrix[16] = {1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, -4.2f, 1.0f};
	const float InitialStarsMatrix[16] = {1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f};

	// Set initial drawing options state
	MovementEnabled = true;
	DepthBufferEnabled = true;
	AnimationEnabled = true;
	BinocularEnabled = false;

	// Move camera
	memcpy(CameraMatrix, InitialCameraMatrix, sizeof(CameraMatrix));

	Mode = TurnMode;
	RotationAmount = 0.02f;
	TranslationAmount = 0.02f;
	ScaleFactor = 0.01f;
	FocalFactor = 2.0f;
	RearClippingPlane = 13.0f;
	NoseDistance = 0.06f;
	OverlayLevel = 1;

	// Initialize stars matrix
	memcpy(StarsMatrix, InitialStarsMatrix, sizeof(StarsMatrix));

	// Put the mouse in the center
	mouse_x = window_width >> 1;
	mouse_y = window_width >> 1;

	GlobalAmbientLight[0] = 0.2f;
	GlobalAmbientLight[1] = 0.2f;
	GlobalAmbientLight[2] = 0.2f;
	GlobalAmbientLight[3] = 1.0f;

	// Read the command-line arguments
	for(LinkedList<CommandArgument*>::Iterator itr = Arguments.begin(); itr != Arguments.end(); ++itr)
	{
		CommandArgument* argument = *itr;
		LinkedList<CommandValue>* values;

		// Read the ambient light arguments
		if(strcmp(argument->GetName(), "L") == 0)
		{
			values = argument->GetValues();
			if(values->size() >= 3)
			{
				LinkedList<CommandValue>::Iterator value_itr = values->begin();
				GlobalAmbientLight[0] = (*value_itr).GetFloat();
				++value_itr;
				GlobalAmbientLight[1] = (*value_itr).GetFloat();
				++value_itr;
				GlobalAmbientLight[2] = (*value_itr).GetFloat();
			}
		}
	}

	// Verify the command-line input
	for(int i = 0; i < 3; ++i)
	{
		if(GlobalAmbientLight[i] < 0.0f)
		{
			GlobalAmbientLight[i] = 0.0f;
		}
		else if(GlobalAmbientLight[i] > 1.0f)
		{
			GlobalAmbientLight[i] = 1.0f;
		}
	}

	for(LinkedList<Object*>::Iterator itr = SceneObjects.begin(); itr != SceneObjects.end(); ++itr)
	{
		(*itr)->ResetAll();
	}
}

// Pause the program for a short time
void SkelWindow::Delay()
{
	// Sleep for 5 milliseconds
#ifdef WIN32
	Sleep(5);
#else
	// Seems like the SGIs don't like to sleep
	usleep(5000);
#endif
}

// Returns true if the specified key is pressed
bool SkelWindow::IsKeyPressed(unsigned char key)
{
	if(key_press[key & 127])
	{
		key_press[key & 127] = false;
		return true;
	}
	else
	{
		return false;
	}
}

// Returns true if the specified special key is pressed
bool SkelWindow::IsSpecialKeyPressed(int key)
{
	if(special_key_press[key & 127])
	{
		special_key_press[key & 127] = false;
		return true;
	}
	else
	{
		return false;
	}
}

// Draw the stars background
void SkelWindow::DisplayStars()
{
	glPushMatrix();
	glMultMatrixf(StarsMatrix);
	glColor4f(0.8f, 0.9f, 1.0f, 1.0f);
	glBegin(GL_POINTS);
	for(int i = 0; i < NumberOfStars; ++i)
	{
		glVertex3fv(StarsPositions[i]);
	}
	glEnd();
	glPopMatrix();
}

// Draw the text overlay
void SkelWindow::DisplayOverlay()
{
	char buffer[256];

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0, 3000.0, 0.0, 3000.0);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	// Choose the color based on the current mode
	if(Mode == TurnMode)
	{
		glColor3f(1.0f, 0.0f, 1.0f);
	}
	else
	{
		glColor3f(1.0f, 1.0f, 0.0f);
	}

	// Place a bullseye dead center
	DisplayString(1485.0f, 1485.0f, "o");

	// Frame rate counter
	sprintf(buffer, "%4.1f fps", Counter.GetFrameRate());
	DisplayString(80.0f, 80.0f, buffer);

	if(OverlayLevel == 1)
	{
		CurrentOverlayPosition = 2840.0f;

		PrintOverlayText("Exit: Esc");

		PrintOverlayText("Binocular: V");

		sprintf(buffer, "%s: Spacebar", (Mode == TurnMode)?"TurnMode":"FlyMode");
		PrintOverlayText(buffer);

		PrintOverlayText("Animation: H");

		PrintOverlayText("Overlay: W");
		
		sprintf(buffer, "Nose Distance (%0.3f): N", NoseDistance);
		PrintOverlayText(buffer);

		sprintf(buffer, "Speed (%0.4f): S", TranslationAmount);
		PrintOverlayText(buffer);

		sprintf(buffer, "Torque (%0.4f): Q", RotationAmount);
		PrintOverlayText(buffer);

		sprintf(buffer, "Front Clipping Plane (%g)", ScaleFactor * FocalFactor);
		PrintOverlayText(buffer);

		sprintf(buffer, "Focal Factor (%g): O", FocalFactor);
		PrintOverlayText(buffer);

		sprintf(buffer, "Scale Factor (%.2g): I", ScaleFactor);
		PrintOverlayText(buffer);

		sprintf(buffer, "Rear Clipping Plane (%.2g): P", RearClippingPlane);
		PrintOverlayText(buffer);

		PrintOverlayText("Reset: Z");

		sprintf(buffer, "Global Ambient Light (%.2g, %.2g, %.2g): A", GlobalAmbientLight[0], GlobalAmbientLight[1], GlobalAmbientLight[2]);
		PrintOverlayText(buffer);
	}

	// Draw all the scene objects
	for(LinkedList<Object*>::Iterator itr = SceneObjects.begin(); itr != SceneObjects.end(); ++itr)
	{
		(*itr)->Draw2DAll();
	}

	glPopMatrix();
}

// Print a string at the specified coordinates on the 2D surface
void SkelWindow::DisplayString(float x, float y, char* text)
{
	char* pChar;

	glRasterPos3f(x, y, 0.0f);
	
	for(pChar = text; *pChar != NULL; ++pChar)
	{
		glutBitmapCharacter(GLUT_BITMAP_9_BY_15, *pChar);
	}
}

// Print the specified text to the overlay (HUD)
void SkelWindow::PrintOverlayText(char* text)
{
	DisplayString(80.0f, CurrentOverlayPosition, text);
	CurrentOverlayPosition -= 80.0f;
}

//////////////////////////////////////////////////////////////////////////////
//
// Light
//
// Sets up an OpenGL light source
//
//////////////////////////////////////////////////////////////////////////////

int Light::NextLightNumber = GL_LIGHT0;

// Default constructor
Light::Light()
{
	LightNumber = NextLightNumber;
	++NextLightNumber;

	Position[0] = 0.0f;
	Position[1] = 0.0f;
	Position[2] = 1.0f;
	Position[3] = 0.0f;

	Direction[0] = 0.0f;
	Direction[1] = 0.0f;
	Direction[2] = -1.0f;

	AmbientColor[0] = 0.0f;
	AmbientColor[1] = 0.0f;
	AmbientColor[2] = 0.0f;
	AmbientColor[3] = 1.0f;

	DiffuseColor[0] = 1.0f;
	DiffuseColor[1] = 1.0f;
	DiffuseColor[2] = 1.0f;
	DiffuseColor[3] = 1.0f;

	SpecularColor[0] = 1.0f;
	SpecularColor[1] = 1.0f;
	SpecularColor[2] = 1.0f;
	SpecularColor[3] = 1.0f;

	SpotExponent = 0.0f;
	SpotCutoff = 180.0f;
	LinearAttenuation = 0.0f;
}

// Enable this light
void Light::Enable()
{
	glLightfv(LightNumber, GL_POSITION, Position);
	glLightfv(LightNumber, GL_SPOT_DIRECTION, Direction);

	glLightfv(LightNumber, GL_AMBIENT, AmbientColor);
	glLightfv(LightNumber, GL_DIFFUSE, DiffuseColor);
	glLightfv(LightNumber, GL_SPECULAR, SpecularColor);

	glLightfv(LightNumber, GL_SPOT_EXPONENT, &SpotExponent);
	glLightfv(LightNumber, GL_SPOT_CUTOFF, &SpotCutoff);
	glLightfv(LightNumber, GL_LINEAR_ATTENUATION, &LinearAttenuation);

	glEnable(LightNumber);
}

// Disable this light
void Light::Disable()
{
	glDisable(LightNumber);
}

void Light::SetPosition(float x, float y, float z, float alpha)
{
	Position[0] = x;
	Position[1] = y;
	Position[2] = z;
	Position[3] = alpha;
}

void Light::SetDirection(float x, float y, float z)
{
	Direction[0] = x;
	Direction[1] = y;
	Direction[2] = z;
}


void Light::SetAmbientColor(float red, float green, float blue, float alpha)
{
	AmbientColor[0] = red;
	AmbientColor[1] = green;
	AmbientColor[2] = blue;
	AmbientColor[3] = alpha;
}


void Light::SetDiffuseColor(float red, float green, float blue, float alpha)
{
	DiffuseColor[0] = red;
	DiffuseColor[1] = green;
	DiffuseColor[2] = blue;
	DiffuseColor[3] = alpha;
}


void Light::SetSpecularColor(float red, float green, float blue, float alpha)
{
	SpecularColor[0] = red;
	SpecularColor[1] = green;
	SpecularColor[2] = blue;
	SpecularColor[3] = alpha;
}


void Light::SetSpotExponent(float exponent)
{
	SpotExponent = exponent;
}


void Light::SetSpotCutoff(float cutoff)
{
	SpotCutoff = cutoff;
}


void Light::SetLinearAttenuation(float attenuation)
{
	LinearAttenuation = attenuation;
}


//////////////////////////////////////////////////////////////////////////////
//
// CommandArgument
//
// Holds a command-line argument passed to the application
//
//////////////////////////////////////////////////////////////////////////////

// Normal constructor
CommandArgument::CommandArgument(char* name)
{
	Name = name;
}

// Adds a value for this argument
void CommandArgument::InsertValue(char* value)
{
	Values.push_back(CommandValue(value));
}

// Return the name of this argument
char* CommandArgument::GetName()
{
	return Name;
}

// Return the values that followed this argument on the command-line
LinkedList<CommandValue>* CommandArgument::GetValues()
{
	return &Values;
}


//////////////////////////////////////////////////////////////////////////////
//
// CommandValue
//
// Holds a command-line argument values passed to the application
//
//////////////////////////////////////////////////////////////////////////////
// Default constructor
CommandValue::CommandValue()
{
	Value = NULL;
}

// Normal constructor
CommandValue::CommandValue(char* value)
{
	Value = value;
}

// Return the value as a string
char* CommandValue::GetString()
{
	return Value;
}

// Return the value as an int
int CommandValue::GetInt()
{
	return atoi(Value);
}

// Return the value as a float
float CommandValue::GetFloat()
{
	return atof(Value);
}


//////////////////////////////////////////////////////////////////////////////
//
// FrameRateCounter
//
// Calculates the application frame rate
//
//////////////////////////////////////////////////////////////////////////////

// Default constructor
FrameRateCounter::FrameRateCounter() : Index(0)
{
	memset(History, 0, sizeof(History));
}

// Schedule the time for the next frame
void FrameRateCounter::ScheduleNextFrame()
{
	// Move the current position in the circular buffer
	if(Index == 8)
	{
		Index = 0;
	}
	else
	{
		++Index;
	}

	// Record this frame's time for the FPS counter
	History[Index] = GetTime();
}

// Returns true if it is time to draw another frame
bool FrameRateCounter::ReadyForNextFrame()
{
#if ENABLE_FRAME_RATE_LIMITER
	// The number of seconds between frames
	const long FrameDelay = 25L;		// A delay of 25 miliseconds is about 40 frames/sec

	// Return true if it has been enough time since the previous frame
	long Now = GetTime();
	return ((History[Index] + FrameDelay) < Now);
#else
	return true;
#endif
}

// Get the current time in miliseconds (since some base time)
long FrameRateCounter::GetTime()
{
#ifdef WIN32
	// Get the current number of clock ticks and convert to miliseconds
	clock_t ticks = clock();
	return ((ticks * 1000) / CLK_TCK);
#else
	// Get the current time converted to miliseconds
	struct timeval value;
	struct timezone zone;
	gettimeofday(&value, &zone);
	return (value.tv_sec * 1000 + value.tv_usec / 1000);
#endif
}

// Calculate the frame rate
float FrameRateCounter::GetFrameRate()
{
	int LastIndex;
	if(Index == 8)
	{
		LastIndex = 0;
	}
	else
	{
		LastIndex = Index + 1;
	}

	// Return the number of frames per second (average for last 8 frames)
	return (8000.0f / (History[Index] - History[LastIndex]));
}

//////////////////////////////////////////////////////////////////////////////
//
// Object
//
// This is the base class for all objects in the scene
//
//////////////////////////////////////////////////////////////////////////////
Object::Object() : AnimationStep(0)
{
	Position[0] = 0.0f;
	Position[1] = 0.0f;
	Position[2] = 0.0f;
	Size[0] = 1.0f;
	Size[1] = 1.0f;
	Size[2] = 1.0f;
	Rotation[0] = 0.0f;
	Rotation[1] = 0.0f;
	Rotation[2] = 0.0f;
}

// Destructor
Object::~Object()
{
	// Delete all children objects
	for(LinkedList<Object*>::Iterator itr = children.begin(); itr != children.end(); ++itr)
	{
		delete *itr;
	}
}

// Add a child object
void Object::InsertChild(Object* object)
{
	children.push_back(object);
}

// Restore the default settings for this object and all its children
void Object::ResetAll()
{
	// Restore this object's default settings
	Reset();

	// Restore the default settings on all child objects
	for(LinkedList<Object*>::Iterator itr = children.begin(); itr != children.end(); ++itr)
	{
		(*itr)->ResetAll();
	}
}

// Draw this object and all its children
void Object::DrawAll()
{
	glPushMatrix();

	glRotatef(Rotation[0], 1.0f, 0.0f, 0.0f);
	glRotatef(Rotation[1], 0.0f, 1.0f, 0.0f);
	glRotatef(Rotation[2], 0.0f, 0.0f, 1.0f);
	glTranslatef(Position[0], Position[1], Position[2]);
	glScalef(Size[0], Size[1], Size[2]);

	// Draw this object
	Draw();

	// Draw all child objects (and their children, etc.)
	for(LinkedList<Object*>::Iterator itr = children.begin(); itr != children.end(); ++itr)
	{
		(*itr)->DrawAll();
	}

	glPopMatrix();
}

// Draw2D for this object and all its children
void Object::Draw2DAll()
{
	glPushMatrix();

	// Draw2D for this object
	Draw2D();

	// Draw2D for all child objects (and their children, etc.)
	for(LinkedList<Object*>::Iterator itr = children.begin(); itr != children.end(); ++itr)
	{
		(*itr)->Draw2DAll();
	}

	glPopMatrix();
}

// Step forward the animation for this object and all its children
void Object::StepAnimationAll()
{
	// Step forward the animation
	StepAnimation();

	// Do it for all child objects (and their children, etc.)
	for(LinkedList<Object*>::Iterator itr = children.begin(); itr != children.end(); ++itr)
	{
		(*itr)->StepAnimationAll();
	}
}

// Set the object position
void Object::SetPosition(float x, float y, float z)
{
	Position[0] = x;
	Position[1] = y;
	Position[2] = z;
}

// Set the object size
void Object::SetSize(float x, float y, float z)
{
	Size[0] = x;
	Size[1] = y;
	Size[2] = z;
}

// Set the object rotation for each axis
void Object::SetRotation(float angle_x, float angle_y, float angle_z)
{
	Rotation[0] = angle_x;
	Rotation[1] = angle_y;
	Rotation[2] = angle_z;
}


//////////////////////////////////////////////////////////////////////////////
//
// Torus
//
//////////////////////////////////////////////////////////////////////////////
// Default constructor
Torus::Torus() : Object()
{
}

void Torus::Reset()
{
	Gap = 1.0f;

	// Read the command-line arguments
	for(LinkedList<CommandArgument*>::Iterator itr = window->Arguments.begin(); itr != window->Arguments.end(); ++itr)
	{
		CommandArgument* argument = *itr;
		LinkedList<CommandValue>* values;

		// Read the ambient light arguments
		if(strcmp(argument->GetName(), "g") == 0)
		{
			values = argument->GetValues();
			if(values->size() > 0)
			{
				Gap = values->front().GetFloat();
			}
		}
	}

	// Verify the user input
	if(Gap < 0.0f)
	{
		Gap = 0.0f;
	}
	else if(Gap > 1.0f)
	{
		Gap = 1.0f;
	}

	AnimationStep = 0;
	DrawFactor = 1.0f;
	NormalVectors = false;
}

// Draw the torus
void Torus::Draw()
{
	const float max_step = 0.2f;
	float distance1, distance2;
	int num_steps1, num_steps2;
	float step_size1, step_size2;
	float red, green, blue, next_green, next_blue;
	float normal_x, normal_y, normal_z;
	float position1, position2, next_position1;

	// Set the material specular reflectiveness and emission
	float specular[] = {1.0f, 1.0f, 1.0f, 1.0f};
	float emission[] = {0.0f, 0.0f, 0.0f, 1.0f};
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
	glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, emission);

	// Make the material shiny
	const float shininess = 30.0f;
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &shininess);

	// Adjust the gap amount
	if(window->IsKeyPressed(IncreaseGapKey))
	{
		Gap += 0.1f;
	}
	if(window->IsKeyPressed(DecreaseGapKey))
	{
		Gap -= 0.1f;
	}

	// Toggle drawing normal vectors
	NormalVectors ^= window->IsKeyPressed(ToggleNormalVectorsKey);

	// figure out how much of the torus we're going to draw
	distance1 = DrawFactor * 2.0f * PI;
	distance2 = DrawFactor * 2.0f * PI;

	// each step should be as big a possible but keeping the surface smooth
	num_steps1 = 1.0f + distance1 / max_step;
	num_steps2 = 1.0f + distance2 / max_step;
	step_size1 = distance1 / num_steps1;
	step_size2 = distance2 / num_steps2;

	// Draw the torus out of strips
	position1 = 0.0f;
	for(int i = 0; i < num_steps1; ++i)
	{
		// the width of the strip
		next_position1 = position1 + Gap * step_size1;

		// Choose some interesting colors
		blue =  1.0f - position1 / distance1;
		green =  0.25f + abs(0.5f - blue);
		next_blue =  1.0f - next_position1 / distance1;
		next_green =  0.25f + abs(0.5f - next_blue);

		// Draw each strip out of triangles
		glBegin(GL_TRIANGLE_STRIP);
		position2 = 0.0f;
		for(int j = 0; j < num_steps2; ++j)
		{
			// radius of unit sphere is also unit normal to the torus
			normal_x = cos(position1) * cos(position2);
			normal_y = sin(position1) * cos(position2);
			normal_z = sin(position2);

			// Choose some interesting colors
			red =  position2 / distance2;

			// Draw the first point
			glNormal3f(normal_x, normal_y, normal_z);
			glColor3f(red, green, blue);
			glVertex3f(cos(position1) + 0.5f * normal_x, sin(position1) + 0.5f * normal_y, 0.5f * normal_z);

			normal_x = cos(next_position1) * cos(position2);
			normal_y = sin(next_position1) * cos(position2);

			// Draw the second point
			glNormal3f(normal_x, normal_y, normal_z);
			glColor3f(red, next_green, next_blue);
			glVertex3f(cos(next_position1) + 0.5f * normal_x, sin(next_position1) + 0.5f * normal_y, 0.5f * normal_z);

			position2 += step_size2;
		}
		glEnd();

		// Draw the normal vectors
		if(NormalVectors == true)
		{
			position2 = 0.0f;
			for(int j = 0; j < num_steps2; ++j)
			{
				// radius of unit sphere is also unit normal to the torus
				normal_x = cos(position1) * cos(position2);
				normal_y = sin(position1) * cos(position2);
				normal_z = sin(position2);

				// Draw the normal vector
				glBegin(GL_LINE_STRIP);
				glColor3f(1,1,1);
				glVertex3f(cos(position1) + 0.5f * normal_x + 0.01f * normal_x, sin(position1) + 0.5f * normal_y + 0.01f * normal_y, 0.5f * normal_z + 0.01f * normal_z);
				glVertex3f(cos(position1) + 0.5f * normal_x + 0.3f * normal_x, sin(position1) + 0.5f * normal_y + 0.3f * normal_y, 0.5f * normal_z + 0.3f * normal_z);
				glEnd();

				position2 += step_size2;
			}
		}

		position1 += step_size1;
	}
}

// Draw 2D items such as messages on the overlay (HUD)
void Torus::Draw2D()
{
	char buffer[32];

	sprintf(buffer, "Torus Gap (%.2g): G", Gap);
	window->PrintOverlayText(buffer);

	window->PrintOverlayText("Torus Normal Vectors: T");
}

// Step forward the animation for the torus
void Torus::StepAnimation()
{
	if(AnimationStep < 150)
	{
		// Shrink
		DrawFactor -= 0.004;
	}
	else if(AnimationStep < 170)
	{
		// Pause
	}
	else if(AnimationStep < 320)
	{
		// Grow
		DrawFactor += 0.004;
	}
	else if(AnimationStep < 350)
	{
		// Pause
	}

	AnimationStep++;
	if(AnimationStep >= 350)
	{
		AnimationStep = 0;
	}
}


//////////////////////////////////////////////////////////////////////////////
//
// Cube
//
//////////////////////////////////////////////////////////////////////////////
// Default constructor
Cube::Cube() : Object()
{
}

// Restore the default settings for the cube
void Cube::Reset()
{
}

// Draw the cube
void Cube::Draw()
{
	// Set the material specular reflectiveness and emission
	float specular[] = {1.0f, 1.0f, 1.0f, 1.0f};
	float emission[] = {0.0f, 0.0f, 0.0f, 1.0f};
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
	glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, emission);

	// Make the material shiny
	const float shininess = 40.0f;
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &shininess);

	// Draw the cube
	glBegin(GL_TRIANGLE_STRIP);
	glColor3f(0.0f, 0.0f, 0.0f);
	glNormal3f(-0.5f, -0.5f, -0.5f);
	glVertex3f(-0.5f, -0.5f, -0.5f);
	glColor3f(1.0f, 1.0f, 0.0f);
	glNormal3f(0.5f, 0.5f, -0.5f);
	glVertex3f(0.5f, 0.5f, -0.5f);
	glColor3f(1.0f, 0.0f, 0.0f);
	glNormal3f(0.5f, -0.5f, -0.5f);
	glVertex3f(0.5f, -0.5f, -0.5f);
	glColor3f(1.0f, 0.0f, 1.0f);
	glNormal3f(0.5f, -0.5f, 0.5f);
	glVertex3f(0.5f, -0.5f, 0.5f);
	glColor3f(0.0f, 0.0f, 0.0f);
	glNormal3f(-0.5f, -0.5f, -0.5f);
	glVertex3f(-0.5f, -0.5f, -0.5f);
	glColor3f(0.0f, 0.0f, 1.0f);
	glNormal3f(-0.5f, -0.5f, 0.5f);
	glVertex3f(-0.5f, -0.5f, 0.5f);
	glColor3f(0.0f, 1.0f, 1.0f);
	glNormal3f(-0.5f, 0.5f, 0.5f);
	glVertex3f(-0.5f, 0.5f, 0.5f);
	glColor3f(1.0f, 0.0f, 1.0f);
	glNormal3f(0.5f, -0.5f, 0.5f);
	glVertex3f(0.5f, -0.5f, 0.5f);
	glColor3f(1.0f, 1.0f, 1.0f);
	glNormal3f(0.5f, 0.5f, 0.5f);
	glVertex3f(0.5f, 0.5f, 0.5f);
	glColor3f(1.0f, 1.0f, 0.0f);
	glNormal3f(0.5f, 0.5f, -0.5f);
	glVertex3f(0.5f, 0.5f, -0.5f);
	glColor3f(0.0f, 1.0f, 1.0f);
	glNormal3f(-0.5f, 0.5f, 0.5f);
	glVertex3f(-0.5f, 0.5f, 0.5f);
	glColor3f(0.0f, 1.0f, 0.0f);
	glNormal3f(-0.5f, 0.5f, -0.5f);
	glVertex3f(-0.5f, 0.5f, -0.5f);
	glColor3f(0.0f, 0.0f, 0.0f);
	glNormal3f(-0.5f, -0.5f, -0.5f);
	glVertex3f(-0.5f, -0.5f, -0.5f);
	glColor3f(1.0f, 1.0f, 0.0f);
	glNormal3f(0.5f, 0.5f, -0.5f);
	glVertex3f(0.5f, 0.5f, -0.5f);
	glEnd();
}

// Draw 2D items such as messages on the overlay (HUD)
void Cube::Draw2D()
{
}

// Step forward the animation for the cube
void Cube::StepAnimation()
{
}


//////////////////////////////////////////////////////////////////////////////
//
// Tetrahedron
//
//////////////////////////////////////////////////////////////////////////////

// Default constructor
Tetrahedron::Tetrahedron() : Object()
{
}

// Restore the default settings for the tetrahedron
void Tetrahedron::Reset()
{
}

// Draw the tetrahedron
void Tetrahedron::Draw()
{
	// Set the material specular reflectiveness and emission
	float specular[] = {1.0f, 1.0f, 1.0f, 1.0f};
	float emission[] = {0.0f, 0.0f, 0.0f, 1.0f};
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
	glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, emission);

	// Make the material shiny
	const float shininess = 40.0f;
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &shininess);

	const float x0 = -0.5f;
	const float x1 = 0.0f;
	const float x2 = 0.5f;
	const float y0 = -0.20412414523193f;
	const float y1 = 0.6123724356958f;
	const float z0 = -0.28867513459482f;
	const float z1 = 0.0f;
	const float z2 = 0.57735026918962f;

	// Draw the tetrahedron
	glBegin(GL_TRIANGLE_STRIP);
	glColor3f(1.0f, 0.0f, 0.0f);
	glNormal3f(x0, y0, z0);
	glVertex3f(x0, y0, z0);
	glColor3f(0.0f, 1.0f, 0.0f);
	glNormal3f(x2, y0, z0);
	glVertex3f(x2, y0, z0);
	glColor3f(0.0f, 0.0f, 1.0f);
	glNormal3f(x1, y0, z2);
	glVertex3f(x1, y0, z2);
	glColor3f(1.0f, 1.0f, 0.0f);
	glNormal3f(x1, y1, z1);
	glVertex3f(x1, y1, z1);
	glColor3f(1.0f, 0.0f, 0.0f);
	glNormal3f(x0, y0, z0);
	glVertex3f(x0, y0, z0);
	glColor3f(0.0f, 1.0f, 0.0f);
	glNormal3f(x2, y0, z0);
	glVertex3f(x2, y0, z0);
	glEnd();
}

// Draw 2D items such as messages on the overlay (HUD)
void Tetrahedron::Draw2D()
{
}

// Step forward the animation for the tetrahedron
void Tetrahedron::StepAnimation()
{
}



