//////////////////////////////////////////////////////////////////////
// Doublely linked list
//
// This templated list object is similar to the STL vector, but with
// much less functionality.
//
// Author:  Michael Fisher
// Email:   linkedlist@fishdawg.com
//
// Copyright (c) 2004 Michael Fisher
//////////////////////////////////////////////////////////////////////

#ifndef NULL
#define NULL 0L
#endif

#ifndef LINKEDLIST_H
#define LINKEDLIST_H

template <class T>
class LinkedList
{
public:
	//////////////////////////////////////////////////////////////////////
	// LinkedListNode class
	//////////////////////////////////////////////////////////////////////
	class LinkedListNode
	{
	public:
		T Element;
		LinkedListNode* Next;
		LinkedListNode* Prev;

		// Default Constructor
		LinkedListNode() : Prev(NULL), Next(NULL)
		{
		}

		LinkedListNode(T element) : Prev(NULL), Next(NULL), Element(element)
		{
		}
	};


	//////////////////////////////////////////////////////////////////////
	// Iterator class
	//////////////////////////////////////////////////////////////////////
	class Iterator
	{
	private:
		friend class LinkedList<T>;
		friend class ConstIterator;
		LinkedListNode* Node;
		LinkedListNode* NextNode;
		LinkedListNode* PrevNode;

		// Normal constructor
		Iterator(LinkedListNode* node) : Node(node)
		{
			NextNode = node->Next;
			PrevNode = node->Prev;
		}

	public:
		// Default constructor
		Iterator() : Node(NULL), NextNode(NULL), PrevNode(NULL)
		{
		}

		// Copy constructor
		Iterator(const Iterator& itr)
		{
			Node = itr.Node;
			NextNode = itr.NextNode;
			PrevNode = itr.PrevNode;
		}

		// Return the element stored at this position in the list
		T& operator*()
		{
			return Node->Element;
		}

		// Return a pointer to the element stored at this position in the list
		T* operator->()
		{
			return &(Node->Element);
		}

		// Move to the next positon in the list
		Iterator& operator++()
		{
			Node = NextNode;
			NextNode = Node->Next;
			PrevNode = Node->Prev;
			return *this;
		}

		// Move to the previous positon in the list
		Iterator& operator--()
		{
			Node = PrevNode;
			NextNode = Node->Next;
			PrevNode = Node->Prev;
			return *this;
		}

		// Move to the next positon in the list
		// (postfix version of operator++)
		Iterator operator++(int)
		{
			Iterator temp = *this;
			Node = NextNode;
			NextNode = Node->Next;
			PrevNode = Node->Prev;
			return temp;
		}

		// Move to the previous positon in the list
		// (postfix version of operator--)
		Iterator operator--(int)
		{
			Iterator temp = *this;
			Node = PrevNode;
			NextNode = Node->Next;
			PrevNode = Node->Prev;
			return temp;
		}

		// Compare two iterators for equality
		int operator==(const Iterator& origVal) const
		{
			return (Node == origVal.Node);
		}

		// Compare two iterators for inequality
		int operator!=(const Iterator& origVal) const
		{
			return (Node != origVal.Node);
		}
	};


	//////////////////////////////////////////////////////////////////////
	// ConstIterator class
	//////////////////////////////////////////////////////////////////////
	class ConstIterator
	{
	private:
		friend class LinkedList<const T>;
		LinkedListNode* Node;

		// Normal constructor
		ConstIterator(LinkedListNode* node) : Node(node)
		{
		}

	public:
		// Default constructor
		ConstIterator() : Node(NULL)
		{
		}

		// Copy constructor
		ConstIterator(const ConstIterator& itr)
		{
			Node = itr.Node;
			NextNode = itr.NextNode;
			PrevNode = itr.PrevNode;
		}

		// Return the element stored at this position in the list
		const T& operator*()
		{
			return Node->Element;
		}

		// Return a pointer to the element stored at this position in the list
		const T* operator->()
		{
			return &(Node->Element);
		}

		// Move to the next positon in the list
		ConstIterator& operator++()
		{
			Node = NextNode;
			NextNode = Node->Next;
			PrevNode = Node->Prev;
			return *this;
		}

		// Move to the previous positon in the list
		ConstIterator& operator--()
		{
			Node = PrevNode;
			NextNode = Node->Next;
			PrevNode = Node->Prev;
			return *this;
		}

		// Move to the next positon in the list
		// (postfix version of operator++)
		ConstIterator operator++(int)
		{
			ConstIterator temp = *this;
			Node = NextNode;
			NextNode = Node->Next;
			PrevNode = Node->Prev;
			return temp;
		}

		// Move to the previous positon in the list
		// (postfix version of operator--)
		ConstIterator operator--(int)
		{
			ConstIterator temp = *this;
			Node = PrevNode;
			NextNode = Node->Next;
			PrevNode = Node->Prev;
			return temp;
		}

		// Compare two iterators for equality
		int operator==(const ConstIterator& origVal) const
		{
			return (Node == origVal.Node);
		}

		// Compare two iterators for inequality
		int operator!=(const ConstIterator& origVal) const
		{
			return (Node != origVal.Node);
		}
	};

private:
	LinkedListNode* DummyNode;
	int Size;

public:
	// Default constructor
	LinkedList() : Size(0)
	{
		DummyNode = new LinkedListNode();

		DummyNode->Next = DummyNode;
		DummyNode->Prev = DummyNode;
	}

	// Constructor: creates a list of the specified number of (default) elements
	LinkedList(int size) : Size(size)
	{
		LinkedListNode* node1;
		LinkedListNode* node2;

		DummyNode = new LinkedListNode();
		DummyNode->Prev = DummyNode;

		node2 = DummyNode;
		for(int i = 0; i < size; ++i)
		{
			node1 = new LinkedListNode();
			node1->Next = node2;
			node2->Prev = node1;
			node2 = node1;
		}
		DummyNode->Next = node2;
	}

	// Constructor: creates a list of the specified number of the specified element
	LinkedList(int size, T element) : Size(size)
	{
		LinkedListNode* node1;
		LinkedListNode* node2;

		DummyNode = new LinkedListNode();
		DummyNode->Prev = DummyNode;

		node2 = DummyNode;
		for(int i = 0; i < size; ++i)
		{
			node1 = new LinkedListNode(element);
			node1->Next = node2;
			node2->Prev = node1;
			node2 = node1;
		}
		DummyNode->Next = node2;
	}

	// Copy constructor
	LinkedList(const LinkedList& origVal)
	{
		LinkedListNode* node1;
		LinkedListNode* node2;

		DummyNode = new LinkedListNode();

		node2 = DummyNode;
		for(LinkedListNode* origNode = origVal.DummyNode->Prev; origNode != origVal.DummyNode; origNode = origNode->Prev)
		{
			node1 = new LinkedListNode(origNode->Element);
			node1->Next = node2;
			node2->Prev = node1;
			node2 = node1;
		}
		node2->Prev = DummyNode;
		DummyNode->Next = node2;

		Size = origVal.Size;
	}

	// Destructor
	virtual ~LinkedList()
	{
		clear();
		delete DummyNode;
	}

	// Assignment operator
	LinkedList& operator=(const LinkedList& origVal)
	{
		if(this != &origVal)
		{
			clear();

			LinkedListNode* node1;
			LinkedListNode* node2;

			node2 = DummyNode;
			for(LinkedListNode* origNode = origVal.DummyNode->Prev; origNode != origVal.DummyNode; origNode = origNode->Prev)
			{
				node1 = new LinkedListNode(origNode->Element);
				node1->Next = node2;
				node2->Prev = node1;
				node2 = node1;
			}
			node2->Prev = DummyNode;
			DummyNode->Next = node2;

			Size = origVal.Size;
		}

		return *this;
	}

	// Return an iterator that points to the first position in the list
	Iterator begin()
	{
		return Iterator(DummyNode->Next);
	}

	// Return an iterator that points to the first position in the list
	ConstIterator begin() const
	{
		return ConstIterator(DummyNode->Next);
	}

	// Return an iterator that points past the last position in the list
	Iterator end()
	{
		return Iterator(DummyNode);
	}

	// Return an iterator that points past the last position in the list
	ConstIterator end() const
	{
		return ConstIterator(DummyNode);
	}

	// Return the element at the first position in the list
	T& front()
	{
		return DummyNode->Next->Element;
	}

	// Return the element at the first position in the list
	const T& front() const
	{
		return DummyNode->Next->Element;
	}

	// Return the element at the last position in the list
	T& back()
	{
		return DummyNode->Prev->Element;
	}

	// Return the element at the last position in the list
	const T& back() const
	{
		return DummyNode->Prev->Element;
	}

	// Insert the element at the last position in the list
	void push_back(T element)
	{
		LinkedListNode* node;

		node = new LinkedListNode(element);
		node->Next = DummyNode;
		node->Prev = DummyNode->Prev;
		DummyNode->Prev->Next = node;
		DummyNode->Prev = node;
		++Size;
	}

	// Remove the element at the last position in the list
	void pop_back()
	{
		if(size != 0)
		{
			LinkedListNode* node = DummyNode->Prev;
			node->Prev->Next = DummyNode;
			DummyNode->Prev = node->Prev;
			delete node;
			--Size;
		}
	}

	// Insert the element at the first position in the list
	void push_front(T element)
	{
		LinkedListNode* node;

		node = new LinkedListNode(element);
		node->Next = DummyNode->Next;
		node->Prev = DummyNode;
		DummyNode->Next->Prev = node;
		DummyNode->Next = node;
		++Size;
	}

	// Remove the element at the first position in the list
	void pop_front()
	{
		if(size != 0)
		{
			LinkedListNode* node = DummyNode->Next;
			node->Next->Prev = DummyNode;
			DummyNode->Next = node->Next;
			delete node;
			--Size;
		}
	}

	// Insert the specified element before the position pointed to by the iterator
	void insert(Iterator itr, T element)
	{
		LinkedListNode* node = new LinkedListNode(element);

		node->Next = itr.Node;
		node->Prev = itr.Node->prev;
		node->Prev->Next = node;
		node->Next->Prev = node;
		++Size;
	}

	// Remove the element in the list at the position pointed to by the iterator
	void erase(Iterator itr)
	{
		LinkedListNode* node = itr.Node;

		if(node != DummyNode)
		{
			node->Prev->Next = node->Next;
			node->Next->Prev = node->Prev;
			delete node;
			--Size;
		}
	}

	// Removes all elements from the list
	void clear()
	{
		LinkedListNode* node1;
		LinkedListNode* node2;

		for(node2 = DummyNode->Prev; node2 != DummyNode; node2 = node1)
		{
			node1 = node2->Prev;
			delete node2;
		}

		DummyNode->Next = DummyNode;
		DummyNode->Prev = DummyNode;

		Size = 0;
	}

	// Return the number of elements in the list
	int size() const
	{
		return Size;
	}

	// Return true if the list is empty
	bool empty() const
	{
		return (Size == 0);
	}
};

#endif



