#ifndef unix
#  define WIN  1
#endif

// WIN LINUX
/*****************************2002*************************************/
/****         Illustrated Analyst by Michael (Brad) Henry          ****/
/****        Built on C++ illiSkel by Matt Woodruff, which         ****/
/****           is a reorganization of skel.c =                    ****/
/****       OpenSkelGlut.c = Noosh97 with CAVE removed             ****/
/****      (C) 1994--2002 Board of Trustees University of Illinois ****/ 
/****      A Model Real-Time Interactive  C++/OpenGL/GLUT  Animator****/
/****    George Francis, Stuart Levy, Glenn Chappell, Chris Hartman****/
/****    illiAnalyst also includes code from Ben Farmer and Doug   ****/
/****        Nachand's Picker.                                     ****/
/****    e-mail  gfrancis@math.uiuc.edu                            ****/
/**********************************************************************/

#include <vector.h>
#include <pair.h>
#include <stdlib.h>
#include <stdio.h>
#include <GL/glut.h> 

#ifdef unix 
#include <sys/time.h>
#endif

#ifdef WIN
#include <sys/timeb.h>
#endif

#include <math.h>

//Analyst includes
#include <fstream.h>
// using namespace std;

#ifdef WIN
#pragma warning (disable:4305)  /* double-to-float  */ 
#pragma warning (disable:4101)  /* unreferenced local variable */ 
#pragma warning (disable:4056)  /* overflow in fp constant arithmetic */ 
#pragma warning (disable:4047) 
#pragma warning (disable:4024) 
#pragma warning (disable:4133) 
#pragma warning (disable:4530) 
#pragma warning (disable:4786) 
#pragma warning (disable:4003) 
#endif

//formerly macros, now typesafe!
template<class TT> inline TT MAX(TT x, TT y){return ((x)<(y))?(y):(x);}
template<class TT> inline TT MIN(TT x, TT y){return ((x)<(y))?(x):(y);}
template<class TT> inline TT CLAMP(TT x, TT u, TT v)
  {return (x<u? u : (x>v ? v: x));}
template<class TT> inline TT ABS(TT u){return ((u)<0 ? -(u): (u));}

//this one is beyond help
#define  FOR(a,b,c)       for((a)=(b);(a)<(c);(a)++)

template<class TT> inline TT DOT(TT* p, TT* q)
  {return((p)[0]*(q)[0]+(p)[1]*(q)[1]+(p)[2]*(q)[2]);}
template<class TT> inline TT NRM(TT* p) {return sqrt(DOT((p),(p)));}
template<class TT> inline void LCPY(TT* u, TT* v)
  {v[0]=u[0]; v[1]=u[1]; v[2]=u[2];}

//We have our own pi, thank you very much!
#ifdef M_PI
#undef M_PI
#endif
const double M_PI = 355.0/113.0;
const double DG = M_PI/180.0;
inline double S(double degrees){return sin(degrees*DG);}
inline double C(double degrees){return cos(degrees*DG);}
#define  random        rand        /* library dependent name  */
#define IFCLICK(K,a){static int ff=1; if(getbutton(K))ff=1-ff; if(ff){a} }
const int MANYSTARS=10000;




/************************** global variables **************************/
int win=1;                 /* used once to choose window size         */
double mysiz,zoom, speed, torq, focal, wfar; /* navigation control variables */
double creep; /* for the convolution movie */
unsigned int PAW,XX,YY,SHIF;  /* used in chaptrack gluttery           */ 
int xwide,yhigh;                /* viewportery width and height       */
int mode,morph,msg,binoc;       /* viewing globals */
#define FLYMODE  (0)         /* yellow: turns around head as center   */ 
#define TURNMODE (1)         /* purple: turns around object center    */ 
//int ii, jj, kk;  float tmp, temp;     /* **causes** gray hairs later  */
GLfloat aff[16], starmat[16], mat[16];   /* OpenGL placement matrices   */ 
double nose;                       /* to eye distance in console       */
char clefs[128];                  /* which keys were pressed last     */
int caveyes = 0;


/****************************************************************
Analyst Globals --  there's a lot of 'em
****************************************************************/
ofstream output;
const char *outfname = NULL;
const double PI = 355.0 / 113.0;
const int UnitStep = 30;           
const double StepWidth = 1.0 / UnitStep;
int friz, drawf, drawg, drawfhat, drawghat, drawfg, drawcontrol,
drawfghat, drawfhatghat, drawcoord, linewidth, selectF, selectG,
stars, scaleconvolution, background, drawcoordtopright;
int numfunctions = 4;
double axis = 1.75;

/*** Info for function F and Fhat ***/
#define fX funcF[ii][1]
#define pX ((2*(fX-Fstart-(Fend-Fstart)/2))/(Fend-Fstart))
#define FUNCTIONF0  1.0 
#define FUNCTIONF1  pow(2.718, (-1.0)*fX*fX)
#define FUNCTIONF2  fX*fX
#define FUNCTIONF3  fX+((1/2)*fX)+2 
const int Fstart = -2;
const int Fend = 2;
const int Fstep = (Fend - Fstart) * UnitStep;
const int Fvert = Fstep + 1;
const int Fhatstart = -2;
const int Fhatend = 2;
const int Fhatstep = (Fhatend - Fhatstart) * UnitStep;
const int Fhatvert = Fhatstep + 1;

/*** Info for function G and Ghat ***/
#define gX funcG[jj][1]
#define qX ((2*(gX-Gstart-(Gend-Gstart)/2))/(Gend-Gstart))
#define FUNCTIONG0  1.0 
#define FUNCTIONG1 pow(2.718, (-1.0)*(gX - 4.0)*(gX - 4.0))
#define FUNCTIONG2  gX*gX
#define FUNCTIONG3  gX+((1/2)*gX)+2
const int Gstart = 2;
const int Gend = 6;
const int Gstep = (Gend - Gstart) * UnitStep;
const int Gvert = Gstep + 1;
const int Ghatstart = -2;  
const int Ghatend = 2;
const int Ghatstep = (Ghatend - Ghatstart) * UnitStep;
const int Ghatvert = Ghatstep + 1;

/*** Info for convolution C and Chat ***/
const int Chatstart = -2;
const int Chatend = 2;
const int Chatstep = (Chatend - Chatstart) * UnitStep;
const int Chatvert = Chatstep + 1;
const int Cvert = Fvert + Gvert - 1;
const int Cstep = Cvert - 1;

/*** Analyst Arrays ***/
double funcF[Fvert][2];
double funcFhat[Fhatvert][3];
double funcG[Gvert][2];
double funcGhat[Ghatvert][3];
double funcFandG[Cvert][2];
double funcFandGhat[Chatvert][3];
double funcFhatGhat[Fhatvert][3];
double funcDiff[Fhatvert][3];

/*** Picking Windows ***/
double pickxmin = -10.0;
double pickxmax = 10.0;
double pickymin = -10.0;
double pickymax = 10.0;

/*** Scaling Factors for the Tranform Window ***/
double MX_3;
double BX_3;
double MY_3;
double BY_3;
double MZ_3;
double BZ_3;

/*** Scaling Factors for the Convolution Window ***/
double CMX;
double CBX;
double CMY;
double CBY;


/****************************************************************
*                                                               *
*			Main illiAnalyst Code Begins Here                   *
*                                                               *
****************************************************************/


/****************************************************************
Analyst Function Definitions
****************************************************************/
void computeAll();
void initialF();
void initialG();
void computeTransform(double func[][2], double funchat[][3], const int 
			  hatstart, const int hatend, const int numstep, 
			  const int numvert);
double integrateReal(double func[][2], double zz, const double stepsize, 
			 const int numstep);
double integrateComplex(double func[][2], double zz, const double stepsize, 
			const int numstep);
void computeConvolution(double funcF[][2], double funcG[][2], double 
				funcFandG[][2]);
double integrateConvolution(double tempC[][2], const int steps);
void computeProduct(double funcFhat[][3], double funcGhat[][3], double 
			funcFhatGhat[][3], const int numvert); 
void computeDifference(double funcFandGhat[][3], double funcFhatGhat[][3],
			   double funcDiff[][3],const int Chatvert, 
			   const int Fhatvert);

void outputFile();
void output2DArray(double func[][2], const int numvert);
void output3DArray(double func[][3], const int numvert);

void drawTransforms();
void drawCoordAxis();
void draw2DArray(double func[][2], const int numvert, double rr, 
				 double gg, double bb, double end, double start, 
				 double axis);
void draw3DArray(double func[][3], const int numvert, double rr,
				 double gg, double bb); 
double lin(double var, double scalar, double offset);
void calculatePickingScale(double &mx, double &bx, double &my, double &by, 
						   double &maxy, double &miny, double &maxx, double &minx);
double lirp(double var, double min, double max, double Min, double Max){
       return lin(var, (Max-Min)/(max-min), (Min*max - min*Max)/(max-min));       
}

/****************************************************************
Analyst Arrays -- 2D Functions and 3D Transforms 
func[][0] = value of f of funcF[][1]
func[][1] = value of x

func[][0] = real coefficient of transform
func[][1] = complex coefficient of transform
func[][2] = value of z
****************************************************************/

/****************************************************************
Analyst Transform Coordinate System  
Purple Line = x-axis = Transform time = Third Array Term
Blue Line = y-axis = Real Coefficient = First Array Term 
Yellow Line = z-axis = Complex Coefficient = Second Array Term
****************************************************************/

/****************************************************************
computeAll -- computes the transforms and convolution
****************************************************************/
void computeAll(){ 

	/*** Function Calls ***/
	computeTransform(funcF, funcFhat, Fhatstart, Fhatend, Fstep, 
		Fhatvert);
	computeTransform(funcG, funcGhat, Ghatstart, Ghatend, Gstep, 
		Ghatvert);
 	computeProduct(funcFhat, funcGhat, funcFhatGhat, Fhatvert);
	computeConvolution(funcF, funcG, funcFandG);
	computeTransform(funcFandG, funcFandGhat, Chatstart, Chatend, 
		Cstep, Chatvert);
	computeDifference(funcFandGhat, funcFhatGhat, funcDiff, 
		Chatvert, Fhatvert);
	
	if(outfname != NULL) {
	    output.open(outfname);
	    outputFile(); 
	    output.close();
	}

}

/****************************************************************
initialF -- set initial values of function F 
****************************************************************/
void initialF(){
	
	int ii;

	/*** Fill F with values ***/
	switch(selectF)
	{
	case 0:
        for (ii = 0; ii < Fvert; ii++){
                funcF[ii][1] = Fstart + StepWidth * ii;
                funcF[ii][0] = FUNCTIONF0;
        }
		break;
	case 1:
        for (ii = 0; ii < Fvert; ii++){
                funcF[ii][1] = Fstart + StepWidth * ii;
                funcF[ii][0] = FUNCTIONF1;
        }
		break;
	case 2:
        for (ii = 0; ii < Fvert; ii++){
                funcF[ii][1] = Fstart + StepWidth * ii;
                funcF[ii][0] = FUNCTIONF2;
        }
		break;
	case 3:
        for (ii = 0; ii < Fvert; ii++){
                funcF[ii][1] = Fstart + StepWidth * ii;
                funcF[ii][0] = FUNCTIONF3;
        }
		break;
	}

}

/****************************************************************
initialG -- set initial values of function G 
****************************************************************/
void initialG(){
	
	int jj;

        /*** Fill G with values ***/
	switch(selectG)
	{
	case 0:
        for (jj = 0; jj < Gvert; jj++){
                funcG[jj][1] = Gstart + StepWidth * jj;
                funcG[jj][0] = FUNCTIONG0;
        }
		break;
	case 1:
        for (jj = 0; jj < Gvert; jj++){
                funcG[jj][1] = Gstart + StepWidth * jj;
                funcG[jj][0] = FUNCTIONG1;
        }
		break;
	case 2:
        for (jj = 0; jj < Gvert; jj++){
                funcG[jj][1] = Gstart + StepWidth * jj;
                funcG[jj][0] = FUNCTIONG2;
        }
		break;
	case 3:
        for (jj = 0; jj < Gvert; jj++){
                funcG[jj][1] = Gstart + StepWidth * jj;
                funcG[jj][0] = FUNCTIONG3;
        }
		break;
	}

}

/****************************************************************
computeTransform -- compute transform from function
****************************************************************/
void computeTransform(double func[][2], double funchat[][3], const int 
			  hatstart, const int hatend, const int numstep, 
			  const int numvert){

	double zz = 0.0;

	for(int jj = 0; jj < numvert; jj++){
		funchat[jj][2] = hatstart + StepWidth * jj;
		
		zz = funchat[jj][2];

		funchat[jj][0] = integrateReal (func, zz, StepWidth, 
			numstep);
		funchat[jj][1] = integrateComplex (func, zz, StepWidth, 
			numstep);

	}
}

/****************************************************************
integrateReal -- real coefficient of transform
****************************************************************/
double integrateReal(double func[][2], double zz, const double 
				 stepsize, const int numstep) {

	double sumleft = 0.0;
	double sumright = 0.0; 
	double templeft = 0.0;
	double tempright = 0.0;

	double leftend = 0.0;
	double rightend = 0.0;
	double trapezoid = 0.0;
	
	/* Integrate first half of computeTransform transform using 
		trapezoidezoidal rule */
	for(int kk = 0; kk < numstep; kk++)
	{	
		templeft = func[kk][0] * cos(2*PI*func[kk][1]*zz);
		sumleft = sumleft + templeft;

		tempright = func[numstep - kk][0] * 
					cos(2*PI*func[numstep - kk][1]*zz);			
		sumright = sumright + tempright;		
	}
	
	leftend = sumleft * stepsize;
	rightend = sumright * stepsize;
	trapezoid = (rightend+leftend) / 2.0;
	
	double integral_value = trapezoid;

	return integral_value;
}

/****************************************************************
integrateComplex -- complex coefficient of transform
****************************************************************/
double integrateComplex(double func[][2], double zz, const double 
				stepsize, const int numstep){

	double sumleft = 0.0;
	double sumright = 0.0; 
	double templeft = 0.0;
	double tempright = 0.0;

	double leftend = 0.0;
	double rightend = 0.0;
	double trapezoid = 0.0;

	/* Integrate second half of computeTransform transform using 
	Trapezoidal rule */
	for(int mm = 0; mm < numstep; mm++)
	{	
		templeft = (-1.0)*func[mm][0] * sin(2*PI*func[mm][1]*zz);
		sumleft = sumleft + templeft;

		tempright = (-1.0)*func[numstep - mm][0] * 
				sin(2*PI*func[numstep - mm][1]*zz);
		sumright = sumright + tempright;		
	}
	
	leftend = sumleft * stepsize;
	rightend = sumright * stepsize;
	trapezoid = (rightend+leftend) / 2.0;
	
	double integral_value = trapezoid;
	
	return integral_value;
}

/****************************************************************
computeProduct -- piecewise product of two transforms
****************************************************************/
void computeProduct(double funcFhat[][3], double funcGhat[][3], 
			double funcFhatGhat[][3], const int numvert){

	for(int kk = 0; kk < numvert; kk++)
	{	
		double realcoef = (funcFhat[kk][0]*funcGhat[kk][0]) - 
					(funcFhat[kk][1]*funcGhat[kk][1]);
		double complexcoef = (funcFhat[kk][1]*funcGhat[kk][0]) + 
					(funcFhat[kk][0]*funcGhat[kk][1]);
		funcFhatGhat[kk][0] = realcoef;
		funcFhatGhat[kk][1] = complexcoef;
		funcFhatGhat[kk][2] = funcFhat[kk][2];		
	}

}

/****************************************************************
computeConvolution -- convolves two functions 
****************************************************************/
void computeConvolution(double funcF[][2], double funcG[][2], double 
						funcFandG[][2]){
	
	int diff = 0;
	int ii,jj,tt;
	int slide = UnitStep / UnitStep;
	int newGstart = (-1)*Gend;
	int newGend = (-1)*Gstart;

	/*** Arrays Specific to this Function ***/
	double tempFandG[Cvert][2];
	double newG[Gvert][2];
	double tempG[Gvert][2];

	/*** Create a copy of funcG as newG ***/
	for (ii = 0; ii < Gvert; ii++){
		for (jj = 0; jj < 2; jj++){
			newG[ii][jj] = funcG[ii][jj];
		}
	}

	/*** Reflect G ***/
	for (ii = 0; ii < Gvert; ii++){
		newG[ii][1] = (-1.0)*newG[ii][1];
	}

	/*** Place the two functions side by side with G on the right ***/
	if (newGstart != Fend && Fend < newGstart){
		diff = Gstart - Fend;
		for (ii = 0; ii < Gvert; ii++){
			newG[ii][1] = newG[ii][1] - diff;
		}
	}
	else if (newGstart != Fend && Fend > newGstart){
		diff = Fend - newGstart;
		for (ii = 0; ii < Gvert; ii++){
			newG[ii][1] = newG[ii][1] + diff;
		}
	}

	/*** Flip the Array so that the Start is Back at the Front ***/
	/*** This is done using a temp array to hold the flip      ***/
	for (ii = 0; ii < Gvert; ii++){
		for (jj = 0; jj < 2; jj++){
			tempG[Gvert - 1 - ii][jj] = newG[ii][jj];
		}
	}

	/*** Copy the temp array back to newG ***/
	for (ii = 0; ii < Gvert; ii++){
		for (jj = 0; jj < 2; jj++){
			newG[ii][jj] = tempG[ii][jj];
		}
	}
	
	/*** Loop to compute value of convolution at a point ***/

	funcFandG[0][0] = 0.0;
	funcFandG[0][1] = 0.0;
	
	int kk = 0;
	int steps = 0;
	for (tt = 1; tt < Cvert; tt++)
	{

		/*** Slide function G to the left by slide ***/
		for (ii = 0; ii < Gvert; ii++){
			if (newG[ii][1] == 0.0 + StepWidth){
				newG[ii][1] = 0.0;
			}
			else{
				newG[ii][1] = newG[ii][1] - (StepWidth * slide);
			}
		}
		
		/*** Create temp array with values that overlap ***/
		kk = 0;
		for (ii = 0; ii < Gvert; ii++){
			for (jj = 0; jj < Fvert; jj++){
				if ((newG[ii][1] <= (funcF[jj][1] + StepWidth/4)) && 
					(newG[ii][1] >= (funcF[jj][1] - StepWidth/4)) &&
					(newG[ii][1] <= Fend)){
					tempFandG[kk][1] = newG[ii][1];
					tempFandG[kk][0] = newG[ii][0] * funcF[jj][0];
					kk++;
				}
			}
		}
		
		steps = kk - 1;
		/*** Compute the integral of the temp array ***/
		funcFandG[tt][0] = integrateConvolution(tempFandG, steps);
		funcFandG[tt][1] = tt * StepWidth;
	}
}

/****************************************************************
integrateConvolution -- compute value of f*g(t)
****************************************************************/
double integrateConvolution(double tempC[][2], const int steps){

	double sumleft = 0.0;
	double sumright = 0.0; 
	double templeft = 0.0;
	double tempright = 0.0;
	double widthleft = 0.0;
	double widthright = 0.0;

	double trapezoid = 0.0;

	/* Integrate current overlap */
	for(int mm = 0; mm < steps; mm++)
	{	
		widthleft = fabs(tempC[mm+1][1] - tempC[mm][1]);
		templeft = tempC[mm][0]*widthleft;
		sumleft = sumleft + templeft;

		widthright = fabs(tempC[steps-mm][1] - tempC[steps-mm-1][1]);
		tempright = tempC[steps-mm-1][0]*widthright;
		sumright = sumright + tempright;		
	}
	
	trapezoid = (sumright + sumleft) / 2.0;
	
	double integral_value = trapezoid;
	
	return integral_value;
}

/****************************************************************
computeDifference -- compute the pointwise difference between 
			FuncFhatGhat and FuncFandGhat
****************************************************************/
void computeDifference(double funcFandGhat[][3], double funcFhatGhat[][3],
					   double funcDiff[][3],const int Chatvert, 
					   const int Fhatvert){
	if (Chatvert != Fhatvert)
		return;
	
	for (int kk = 0; kk < Chatvert; kk++)
	{
		funcDiff[kk][0] = fabs(funcFandGhat[kk][0] - funcFhatGhat[kk][0]);
		funcDiff[kk][1] = fabs(funcFandGhat[kk][1] - funcFhatGhat[kk][1]);
		funcDiff[kk][2] = funcFhatGhat[kk][2];
	}

}

/****************************************************************
outputFile -- output arrays to a file
****************************************************************/
void outputFile(){

	output << "Function F" << endl;
	output2DArray(funcF, Fvert);
	
	output << endl << "Function G" << endl;
	output2DArray(funcG, Gvert);
	
	output << endl << "F convolved with G" << endl;
	output2DArray(funcFandG, Cvert);
	
	output << endl << "Function F transform" << endl;
	output3DArray(funcFhat, Fhatvert);
	
	output << endl << "Function G transform" << endl;
	output3DArray(funcGhat, Ghatvert);
	
	output << endl << "F transform times G transform" << endl;
	output3DArray(funcFhatGhat, Fhatvert);
	
	output << endl << "F convolved with G transform" << endl;
	output3DArray(funcFandGhat, Fhatvert);
	
	output << endl << "Difference between funcFandGhat and funcFhatGhat" << endl;
	output3DArray(funcDiff, Fhatvert);

}

/****************************************************************
output2DArray -- output the values in a 2D array to a file
****************************************************************/
void output2DArray(double func[][2], const int numvert){
	
	int jj = 0.0;
	int kk = 0.0;
	
	for (jj = 0; jj < numvert; jj++){
		for (kk = 0; kk < 2; kk++){
			output << func[jj][kk] << " ";
		}
		output << "\n";
	}

}

/****************************************************************
output3DArray -- output the values in a 3D array to a file
****************************************************************/
void output3DArray(double func[][3], const int numvert){

	int jj = 0.0;
	int kk = 0.0;
	
	for (jj = 0; jj < numvert; jj++){
		for (kk = 0; kk < 3; kk++){
			output << func[jj][kk] << " ";
		}
		output << "\n";
	}

}

/****************************************************************
compute3DScale -- draw the coordinate axis 
****************************************************************/
void compute3DScale(void){

//LIRP Code for Transform Coordinate Axes
	double xmax=-1000, xmin=1000, ymax=-1000, ymin=1000, zmax=-1000, zmin=1000,
		Xmax, Xmin, Ymax, Ymin, Zmax, Zmin;

	Xmin = (-0.5)*axis;
	Xmax = (0.5)*axis;
	Ymin = (-0.5)*axis;
	Ymax = (0.5)*axis;
	Zmin = (-0.5)*axis;
	Zmax = (0.5)*axis;
	
	int jj;
	for(jj = 0; jj < Fhatvert; jj++){
		if(funcFhat[jj][2] < xmin){
			xmin = funcFhat[jj][2];
		}
		if(funcFhat[jj][2] > xmax){
			xmax = funcFhat[jj][2];
		}
		if(funcFhat[jj][0] < ymin){
			ymin = funcFhat[jj][0];
		}
		if(funcFhat[jj][0] > ymax){
			ymax = funcFhat[jj][0];
		}
		if(funcFhat[jj][1] < zmin){
			zmin = funcFhat[jj][1];
		}
		if(funcFhat[jj][1] > zmax){
			zmax = funcFhat[jj][1];
		}
	}

	for(jj = 0; jj < Ghatvert; jj++){
		if(funcGhat[jj][2] < xmin){
			xmin = funcGhat[jj][2];
		}
		if(funcGhat[jj][2] > xmax){
			xmax = funcGhat[jj][2];
		}
		if(funcGhat[jj][0] < ymin){
			ymin = funcGhat[jj][0];
		}
		if(funcGhat[jj][0] > ymax){
			ymax = funcGhat[jj][0];
		}
		if(funcGhat[jj][1] < zmin){
			zmin = funcGhat[jj][1];
		}
		if(funcGhat[jj][1] > zmax){
			zmax = funcGhat[jj][1];
		}
	}

	for(jj = 0; jj < Fhatvert; jj++){
		if(funcFhatGhat[jj][2] < xmin){
			xmin = funcFhatGhat[jj][2];
		}
		if(funcFhatGhat[jj][2] > xmax){
			xmax = funcFhatGhat[jj][2];
		}
		if(funcFhatGhat[jj][0] < ymin){
			ymin = funcFhatGhat[jj][0];
		}
		if(funcFhatGhat[jj][0] > ymax){
			ymax = funcFhatGhat[jj][0];
		}
		if(funcFhatGhat[jj][1] < zmin){
			zmin = funcFhatGhat[jj][1];
		}
		if(funcFhatGhat[jj][1] > zmax){
			zmax = funcFhatGhat[jj][1];
		}
	}

	for(jj = 0; jj < Chatvert; jj++){
		if(funcFandGhat[jj][2] < xmin){
			xmin = funcFandGhat[jj][2];
		}
		if(funcFandGhat[jj][2] > xmax){
			xmax = funcFandGhat[jj][2];
		}
		if(funcFandGhat[jj][0] < ymin){
			ymin = funcFandGhat[jj][0];
		}
		if(funcFandGhat[jj][0] > ymax){
			ymax = funcFandGhat[jj][0];
		}
		if(funcFandGhat[jj][1] < zmin){
			zmin = funcFandGhat[jj][1];
		}
		if(funcFandGhat[jj][1] > zmax){
			zmax = funcFandGhat[jj][1];
		}
	}

	MX_3 = (Xmax - Xmin)/(xmax - xmin);
	BX_3 = (xmax*Xmin - xmin*Xmax)/(xmax - xmin);

	MY_3 = (Ymax - Ymin)/(ymax - ymin);

	double shifty = (0.5)*axis - fabs(ymin*MY_3);	

	//BY_3 = (ymax*Ymin - ymin*Ymax)/(ymax - ymin);	
	BY_3 = (ymax*Ymin - ymin*Ymax)/(ymax - ymin) + shifty;	

	MZ_3 = (Zmax - Zmin)/(zmax - zmin);
	BZ_3 = (zmax*Zmin - zmin*Zmax)/(zmax - zmin);	

}
/****************************************************************
drawCoordAxis -- draw the coordinate axis 
****************************************************************/
void drawCoordAxis(){

   glColor3f(0.0, 1.0, 1.0);

   glBegin(GL_LINES);
      glVertex3f(0.0, (-1.0)*axis, 0.0);
      glVertex3f(0.0, axis, 0.0);

      glColor3f(0.6, 0.2, 4.0);
      glVertex3f((-1.0)*axis, 0.0, 0.0);
      glVertex3f(axis, 0.0, 0.0);
      
      glColor3f(1.0, 1.0, 0.0);
      glVertex3f(0.0, 0.0, (-1.0)*axis);
      glVertex3f(0.0, 0.0, axis);
   glEnd();

}

/**********************************************************************/
/* drawConvolution - draws the convolution to the lower right         */
/**********************************************************************/
void drawConvolution(void){
      int viewport[4];
      glViewport(xwide/2,0,xwide/2,yhigh/2); /* UR quadrant viewport */
      glMatrixMode(GL_PROJECTION); glLoadIdentity();
        glOrtho(-6, 6, -6, 6, -6, 6);
      glMatrixMode(GL_MODELVIEW);
        glPushMatrix(); glLoadIdentity();
  //      glMultMatrixf(slatemat); /* MODELVIEW matrix for this viewport*/
       glScalef(.5,.5,.5);
       glTranslatef(-.5,0,0);

 glColor3f(0.0, 0.5, 1.0);
 double cx1, cy1;
 if (scaleconvolution==1){
//LIRP Code for controlF
	double cxmax, cxmin, cymax=0, cymin=0, cXmax, 
		cXmin, cYmax, cYmin;
	cxmin = funcFandG[0][1];
	cxmax = funcFandG[Cstep][1];
	cXmin = pickxmin;
	cXmax = pickxmax;
	cYmin = pickymin;
	cYmax = pickymax;
	for(int jj = 0; jj < Cvert; jj++){
		if(funcFandG[jj][0] < cymin){
			cymin = funcFandG[jj][0];
		}
		if(funcFandG[jj][0] > cymax){
			cymax = funcFandG[jj][0];
		}
	}
	CMX = (cXmax - cXmin)/(cxmax - cxmin);
	CBX = (cxmax*cXmin - cxmin*cXmax)/(cxmax - cxmin);
	
	CMY = (cYmax - cYmin)/(cymax - cymin);
	CBY = (cymax*cYmin - cymin*cYmax)/(cymax - cymin);
 }

/* THERE */
  int max = (int)lirp(creep,0, -3.34 , 0, (double)Cvert);
  glColor3f(1.0, 0.3, 0.3);
  glBegin(GL_LINES);
  for(int kk = 0; kk < max; kk++){ 
	cx1 = funcFandG[kk][1]; 
	cy1 = funcFandG[kk][0];
	glVertex3f(lin(cx1,CMX,CBX),  pickymin , 0);
	glVertex3f(lin(cx1,CMX,CBX), lin(cy1,CMY,CBY), 0);
   }
   glEnd();

   glColor3f(0.,0.5,1.);
   glBegin(GL_LINE_STRIP);
      for(int ii=0; ii<Cvert; ii++){
		 cx1 = funcFandG[ii][1];
		 cy1 = funcFandG[ii][0];
         glVertex3f(lin(cx1,CMX,CBX), lin(cy1,CMY,CBY), 0);
      }
   glEnd();

   glMatrixMode(GL_MODELVIEW);
   glPopMatrix();  /* pop the matrix on the MODELVIEW stack */
}

/**********************************************************************/

/****************************************************************
draw2DArray -- draw the 2D array on the screen
****************************************************************/
//This Code is no longer called by the program
void draw2DArray(double func[][2], const int numvert, double rr, 
				 double gg, double bb, double end, double start, 
				 double axis){

//LIRP Code
	double x1, y1, z1, x2, y2, z2;
	double xmax, xmin, ymax=0, ymin=0, Xmax, 
		Xmin, Ymax, Ymin, mx, bx, my, by;
	xmin = start;
	xmax = end;
	Xmin = 0.0;
	Xmax = .90*axis;
	Ymin = 0.0;
	Ymax = .90*axis;
	for(int jj = 0; jj < numvert; jj++){
		if(func[jj][0] < ymin){
			ymin = func[jj][0];
		}
		if(func[jj][0] > ymax){
			ymax = func[jj][0];
		}
	}
	mx = (Xmax - Xmin)/(xmax - xmin);
	bx = (xmax*Xmin - xmin*Xmax)/(xmax - xmin);
	
	my = (Ymax - Ymin)/(ymax - ymin);
	by = (ymax*Ymin - ymin*Ymax)/(ymax - ymin);

	glColor3f(rr, gg, bb);

	glBegin(GL_LINE_STRIP);
	for(int ii = 0; ii < numvert; ii++){
		 x1 = func[ii][1];
		 y1 = func[ii][0];
		 glVertex3d(lin(x1,mx,bx), lin(y1,my,by), 0.0);
	}
	glEnd();

}

/****************************************************************
draw3DArray -- draw the 3D array on the screen
****************************************************************/
void draw3DArray(double func[][3], const int numvert, double rr, 
				 double gg, double bb){
   
    glColor3f(rr, gg, bb);
   
	glBegin(GL_LINE_STRIP);
      for(int ii=0; ii<numvert; ii++){
		 double x1 = func[ii][2];
		 double y1 = func[ii][0];
		 double z1 = func[ii][1];
         glVertex3f(lin(x1,MX_3,BX_3), lin(y1,MY_3, BY_3), lin(z1,MZ_3,BZ_3));
      }
    glEnd();

}

/****************************************************************
drawTransforms -- output arrays to the Screen
****************************************************************/
void drawTransforms(){
	
	if(drawfhat == 1){	
		draw3DArray(funcFhat, Fhatvert, 1.0, 1.0, 1.0);
		//Color is Dark Blue
	}

	if(drawghat == 1){	
		draw3DArray(funcGhat, Ghatvert, 1.0, 0.0, 1.0);
		//Color is Gold
	}

	if(drawfhatghat == 1){	
		draw3DArray(funcFhatGhat, Fhatvert, 1.0, 0.5, 0.0);
		//Color is White 
	}
	
	if(drawfghat == 1){	
		draw3DArray(funcFandGhat, Chatvert, 0.5, 1.0, 0.0);
		//Color is Green 
	}

}

/****************************************************************
lin -- linear interpolation to scale the array
****************************************************************/
double lin(double var, double scalar, double offset){
	double newcoord = var*scalar + offset;
	return newcoord;
}

/****************************************************************
*                                                               *
*			Main illiAnalyst Code Ends Here                     *
*                                                               *
****************************************************************/


/**********************************************************************/
/* Begin Ben Farmer's Picking Code + illiAnalyst modifications        */
/**********************************************************************/

/*********************** picking *************************************/
#define BUFSIZE 512
#define CUBELET  .55
#define CONTROLPTS 13

/* New Code Added by MHenry */
double pickF[Fvert][2];
double controlF[CONTROLPTS][3];
int splineF = Fvert/(CONTROLPTS-3);

double pickG[Gvert][2];
double controlG[CONTROLPTS][3];
int splineG = Gvert/(CONTROLPTS-3);
int pickedcube[2];

float slatemat[16];
int dragmode;
int pCubeId;  /* we picked cube number Id */
int mousePosition[3];
/**********************************************************************/
/* calculatecardinalspline - computes the cardinal spline for a set   */
/*     of 4 control points and a given value of paramT in one         */
/*     in one dimension. Returns the value at the given value of      */
/*     paramT.                                                        */
/**********************************************************************/
float calculatecardinalspline(float P0, float P1, float P2, float P3, float paramT){
   float TEMP[4];
    /* {-1,  1, -1,  1} {P0} {TEMP[0]} */
    /* { 2, -2,  1, -1} {P1} {TEMP[1]} */
    /* {-1,  0,  1,  0}*{P2}={TEMP[2]} */
    /* { 0,  1,  0,  0} {P3} {TEMP[3]} */
    /* Cardinal Matrix K times the 4 control points */

    /*fprintf(stderr,"t = %0.20f \n", paramT);*/

   TEMP[0] = -P0 + P1 - P2 + P3;
   TEMP[1] = 2*P0 - 2*P1 + P2 - P3;
   TEMP[2] = -P0 + P2;
   TEMP[3] = P1;

    /* {t^3 t^2 t 1}*{TEMP[0]} */
    /*               {TEMP[1]} */
    /*               {TEMP[2]} */
    /*               {TEMP[3]} */

    return paramT*paramT*paramT*TEMP[0] + paramT*paramT*TEMP[1] + paramT*TEMP[2] + TEMP[3];
}
/**********************************************************************/
/* calculatedevelopmentF - calculates the development function.        */
/**********************************************************************/
void calculatedevelopmentF(void){
   float tParam;
   int ii, jj;
   for(ii=0; ii<(CONTROLPTS-3); ii++){
      for(jj=0; jj<splineF; jj++){
         tParam = ((float)jj)/((float)splineF);

         pickF[ii*splineF+jj][0] \
          = calculatecardinalspline(controlF[ii][0], controlF[ii+1][0],\
          controlF[ii+2][0], controlF[ii+3][0], tParam);
      }
   }

   for(ii=0; ii<Fvert; ii++){
	   funcF[ii][0] = pickF[ii][0];
   }
}

/**********************************************************************/
/* calculatedevelopmentG - calculates the development function.        */
/**********************************************************************/
void calculatedevelopmentG(void){
   float tParam;
   int ii, jj;

   for(ii=0; ii<(CONTROLPTS-3); ii++){
      for(jj=0; jj<splineG; jj++){
         tParam = ((float)jj)/((float)splineG);

         pickG[ii*splineG+jj][0] \
          = calculatecardinalspline(controlG[ii][0], controlG[ii+1][0],\
          controlG[ii+2][0], controlG[ii+3][0], tParam);
      }
   }

   for(ii=0; ii<Gvert; ii++){
	   funcG[ii][0] = pickG[ii][0];
   }
}
/**********************************************************************/
/* initpoints - intializes the control points and the points in the   */
/*      development.                                                  */
/**********************************************************************/
void initpoints(void){ 

//Initialize pickG and controlG
   int step = Fvert/(CONTROLPTS-3);
   int mod;
   int ii,jj;

   //Set starting control point to function start
   controlF[0][0] = funcF[0][0];
   controlF[0][1] = funcF[0][1];
   controlF[0][2] = 0; 
   
   //Set ending control point to function end
   controlF[CONTROLPTS-1][0] = funcF[Fvert-1][0];
   controlF[CONTROLPTS-1][1] = funcF[Fvert-1][1];
   controlF[CONTROLPTS-1][2] = 0;

   ii=1;
   
   for(jj=0; jj < Fvert; jj++){  
	   mod = jj%step;
	   if(mod == 0 && ii < CONTROLPTS-1){
		  controlF[ii][0] = funcF[jj][0];
		  controlF[ii][1] = funcF[jj][1];
		  controlF[ii][2] = 0;
		  ii++;
	   }
   }

   for(ii=0; ii<Fvert; ii++){
	  pickF[ii][0] = funcF[ii][0];
      pickF[ii][1] = funcF[ii][1];
   }

//Initialize pickG and controlG
   step = Gvert/(CONTROLPTS-3);

   //Set starting control point to function start
   controlG[0][0] = funcG[0][0];
   controlG[0][1] = funcG[0][1];
   controlG[0][2] = 0; 
   
   //Set ending control point to function end
   controlG[CONTROLPTS-1][0] = funcG[Fvert-1][0];
   controlG[CONTROLPTS-1][1] = funcG[Fvert-1][1];
   controlG[CONTROLPTS-1][2] = 0;

   ii=1;
   
   for(jj=0; jj < Gvert; jj++){  
	   mod = jj%step;
	   if(mod == 0 && ii < CONTROLPTS-1){
		  controlG[ii][0] = funcG[jj][0];
		  controlG[ii][1] = funcG[jj][1];
		  controlG[ii][2] = 0;
		  ii++;
	   }
   }

   for(ii=0; ii<Gvert; ii++){
	  pickG[ii][0] = funcG[ii][0];
      pickG[ii][1] = funcG[ii][1];
   }

}
/**********************************************************************/

/**********************************************************************/
/* processhits - decides which object is to be the picked object      */
/**********************************************************************/
void processhits(GLint numbHits, GLuint buffer[]){
   unsigned int  ii, j;
   GLuint names, *ptr ;
   unsigned int whichgroup;
   GLuint biggestZ, currentZ, id;
   biggestZ = currentZ = whichgroup = id = 0;
   
   if(dragmode){ return;}  /* if we have annother point selected */

   if(numbHits!=0){      /* if we have a hit */
      /* we know that if each object has exactly 1 name  */
      /* @ 5*i we have the number of names for the ith hit */
      /* @ 5*i+1 we have the minZ vaule of the ith hit   */
      /* @ 5*i+2 we have the maxZ vaule of the ith hit   */
      /* @ 5*i+3 we have which set the object belongs to */
      /* @ 5*i+4 we have the name of the object we hit   */
      /* I ASSUME EACH OBJECT HAS EXACTLY 2 NAMES        */

      /* compare the minZ values */
      biggestZ = buffer[1];  
      whichgroup = buffer[3];
      id = buffer[4];
      for(ii=0; ii<numbHits; ii++){
         if(biggestZ<buffer[ii*5+1]){
            biggestZ = buffer[ii*5+1];
            whichgroup = buffer[ii*5+3];
            id = buffer[ii*5+4];
         }
      }

      pickedcube[0] = whichgroup;
      pickedcube[1] = id;
      /*fprintf(stderr,"There was a hit \n");*/
   }else{
      pickedcube[0] = -1;
      pickedcube[1] = -1;
   }

}

/**********************************************************************/
/* dragtrack - the picking tracker                                    */
/**********************************************************************/
void dragtrack(int but, int xx, int yy, int shif){
        float dx,dy,dz, vv[3];
        dx =torq*(float)(xx-mousePosition[0]);  /*displacement of mouse*/
        dy =(-1.0)*torq*(float)(yy-mousePosition[1]);

        mousePosition[0]=xx;
        mousePosition[1]=yy;
        mousePosition[2]=0;

        /* move the control point */
        if(dragmode){
           if(pickedcube[0] == 1){ /* we picked a cube in controlF */
              controlF[ pickedcube[1] ][0] += dy;
              calculatedevelopmentF(); 
		  	  computeAll();
           }else if(pickedcube[0] == 2){ /* we picked a cube in controlG */
              controlG[ pickedcube[1] ][0] += dy;
              calculatedevelopmentG(); 
		  	  computeAll();
           }else{} 
        }
}
/**********************************************************************/

/**********************************************************************/
/* calculatePickingScale - calculates the scaling factors for the picking*/
/**********************************************************************/
void calculatePickingScale(double &mx, double &bx, double &my, double &by, 
						   double &maxy, double &miny, double &maxx, double &minx){

	double fx1, fy1, gx1, gy1;
	double xmin, xmax, ymax=0, ymin=0, Xmax, 
		Xmin, Ymax, Ymin;
	xmin = min(Fstart, Gstart);
	xmax = max(Fend, Gend);
	Xmin = pickxmin;
	Xmax = pickxmax;
	Ymin = pickymin;
	Ymax = pickymax;
	int jj;
	for(jj = 0; jj < Fvert; jj++){
		if(pickF[jj][0] < ymin){
			ymin = pickF[jj][0];
		}
		if(pickF[jj][0] > ymax){
			ymax = pickF[jj][0];
		}
	}
	for(jj = 0; jj < Gvert; jj++){
		if(pickG[jj][0] < ymin){
			ymin = pickG[jj][0];
		}
		if(pickG[jj][0] > ymax){
			ymax = pickG[jj][0];
		}
	}
	mx = (Xmax - Xmin)/(xmax - xmin);
	bx = (xmax*Xmin - xmin*Xmax)/(xmax - xmin);
	
	my = (Ymax - Ymin)/(ymax - ymin);
	by = (ymax*Ymin - ymin*Ymax)/(ymax - ymin);
	
	//Change the appropriate variables from the calling function
	miny = ymin;
	maxy = ymax;
	minx = xmin;
	maxx = xmax;

}

/**********************************************************************/
/* drawdevelopment - draws the top development function              */
/**********************************************************************/
void drawdevelopment(void){
   glColor3f(1.0, 1.0, 1.0);
   int ii;

	double mx = 0.0;
	double bx = 0.0;
	double my = 0.0;
	double by = 0.0;
	double miny=0, maxy=0, maxx=0, minx=0;

	calculatePickingScale(mx, bx, my, by, maxy, miny, maxx, minx);

    if(drawcoordtopright==1){/* Draw a Coordinate Axis in Upper Right */
	   glColor3f(0.3, 0.3, 0.3);
	   glBegin(GL_LINES);
			glVertex3f(lin(minx,mx,bx), lin(0.0,my,by), 0);
			glVertex3f(lin(maxx,mx,bx), lin(0.0,my,by), 0);
	   glEnd();
	   glBegin(GL_LINES);
			glVertex3f(lin(0.0,mx,bx), lin(miny,my,by), 0);
			glVertex3f(lin(0.0,mx,bx), lin(maxy,my,by), 0);
	   glEnd();
	}
	
	double gx1, gy1;
	glPushMatrix( );        
	glTranslatef(creep, 0,0);
		if(drawg==1){ /* then right function is drawn */
		   glColor3f(1.0, 0.0, 1.0);
		   glBegin(GL_LINES);
				glVertex3f(lin(pickG[0][1],mx,bx), lin(0.0,my,by), 0);
				glVertex3f(lin(pickG[0][1],mx,bx), lin(pickG[0][0],my,by), 0);
		   glEnd();
		   glBegin(GL_LINE_STRIP);
			  for(ii=0; ii<Gvert; ii++){
				 gx1 = pickG[ii][1];
				 gy1 = pickG[ii][0];
				 glVertex3f(lin(gx1,mx,bx), lin(gy1,my,by), 0);
			  }
		   glEnd();
		   glBegin(GL_LINES);
				glVertex3f(lin(pickG[Gvert-1][1],mx,bx), lin(pickG[Gvert-1][0],my,by), 0);
				glVertex3f(lin(pickG[Gvert-1][1],mx,bx), lin(0.0,my,by), 0);
		   glEnd();
		} /* end draw G */
	glPopMatrix();

	double fx1, fy1;
	if(drawf==1){  /* then left function is drawn */
	   glColor3f(1., 1., 1.);
	   glBegin(GL_LINES);
			glVertex3f(lin(pickF[0][1],mx,bx), lin(0.0,my,by), 0);
			glVertex3f(lin(pickF[0][1],mx,bx), lin(pickF[0][0],my,by), 0);
	   glEnd();
	   glBegin(GL_LINE_STRIP);
		  for(ii=0; ii<Fvert; ii++){
			 fx1 = pickF[ii][1];
			 fy1 = pickF[ii][0];
			 glVertex3f(lin(fx1,mx,bx), lin(fy1,my,by), 0);
		  }
	   glEnd();
	   glBegin(GL_LINES);
			glVertex3f(lin(pickF[Fvert-1][1],mx,bx), lin(pickF[Fvert-1][0],my,by), 0);
			glVertex3f(lin(pickF[Fvert-1][1],mx,bx), lin(0.0,my,by), 0);
	   glEnd();
	}

} /* end drawdevelopment */

/**********************************************************************/
/* drawcontrolpoints - draws the control points to the screen        */
/**********************************************************************/
void drawcontrolpoints(GLenum mode){
    glColor3f(1.0, 0.5, 0.0);

	double mx = 0.0;
	double bx = 0.0;
	double my = 0.0;
	double by = 0.0;
	double miny=0, maxy=0, maxx=0, minx=0;

	calculatePickingScale(mx, bx, my, by, maxy, miny, maxx, minx);

	double fx1, fy1;
    glMatrixMode(GL_MODELVIEW);
    glLoadName(1);
    glPushName(0);
    int ii;
    for(ii=0; ii<CONTROLPTS; ii++){
       if(mode == GL_SELECT){
          glLoadName(ii);
       }
       glPushMatrix();
	   	  fx1 = controlF[ii][1];
		  fy1 = controlF[ii][0];
          glTranslatef(lin(fx1,mx,bx),lin(fy1,my,by),0);
          glutSolidCube(CUBELET);
       glPopMatrix();
    }
    glPopName();

    glColor3f(0.3, 0.4, 0.9);

	double gx1, gy1;
    glMatrixMode(GL_MODELVIEW);
    glLoadName(2);
    glPushName(0);
    for(ii=0; ii<CONTROLPTS; ii++){
       if(mode == GL_SELECT){
          glLoadName(ii);
       }
       glPushMatrix();
	   	  gx1 = controlG[ii][1];
		  gy1 = controlG[ii][0];
          glTranslatef(lin(gx1,mx,bx),lin(gy1,my,by),0);
          glutSolidCube(CUBELET);
       glPopMatrix();
    }
    glPopName();
}
/**********************************************************************/
/* sketchslate - does the drawing for the slate                       */
/**********************************************************************/
void sketchslate(GLenum rendermode){
    if(drawcontrol==1 && drawf==1 && drawg==1){
		drawcontrolpoints(rendermode);
	}
    if(rendermode==GL_RENDER){
		drawdevelopment();
    }
}
/**********************************************************************/
/**********************************************************************/
/* drawslate - draws the slate and does the picking from it           */
/**********************************************************************/
void drawslate(void){
      int hits; GLuint selectBuf[BUFSIZE]; int viewport[4];
      float oldProj[16]; /* pushed projection matrix during picking */
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix(); glLoadIdentity();
        glMultMatrixf(slatemat); /* MODELVIEW matrix for this viewport*/
        /***SLATE1***/
        glViewport(xwide/2,yhigh/2,xwide/2,yhigh/2); /* UR quadrant viewport */
        glOrtho(-6, 6, -6, 6, -6, 6);


        /*Splice this code into a prog that you want picking */
           glGetIntegerv(GL_VIEWPORT,viewport); /* store the viewport */
           glGetFloatv(GL_PROJECTION_MATRIX, oldProj); /* store the proj mat */
           glMatrixMode(GL_PROJECTION);
           glPushMatrix(); glLoadIdentity();
              glSelectBuffer(BUFSIZE, selectBuf);
              (void) glRenderMode(GL_SELECT);
              glInitNames();
              glPushName(0);
              gluPickMatrix((GLdouble) XX, (GLdouble) (yhigh-YY), 5.0,5.0,viewport);
              glMultMatrixf(oldProj); /* restores the projection mat we want */
                 /* Draw Things for picking */
                 sketchslate(GL_SELECT);

              hits = glRenderMode(GL_RENDER);
              processhits(hits, selectBuf);
              dragtrack(PAW,XX,YY,SHIF);
           glMatrixMode(GL_PROJECTION);
           glPopMatrix(); /* pop the picking PROJECTION mat */
        /* end of Splice */

        sketchslate(GL_RENDER);
      glMatrixMode(GL_MODELVIEW);
      glPopMatrix();  /* pop the matrix on the MODELVIEW stack */
}
/***********End*Slate*********************/

/**********************************************************************/
/* End of Ben Farmer's Picking Code + illiAnalyst modifications       */
/**********************************************************************/


/**********************************************************************
 * IlliLuminator 
 * illiLight plus illiPaint.
 * Illumination brings light and color.
 **********************************************************************/
class IlliLuminator
{
  public:
  IlliLuminator();
  void initialize(double amb, double pwr, double lux[3]);
  void increasePwr(double multiple);
  void increaseAmb(double multiple);
  void setLux(double newlux[3]);
  double getAmb();
  double getPwr();

/**********************************************************************
 * transformLux
 * multiply the lux vector by the affine matrix to get the luxx 
 * vector
 **********************************************************************/
  void transformLux(GLfloat aff[16])
  {
    for(int ii=0; ii<3; ii++)
    {
      luxx[ii]=0;
      for(int jj=0; jj<3; jj++)
      {
        luxx[ii] += aff[ii*4+jj]*lux[jj];
      }
    }
  }

/**********************************************************************
 * illuminate maps a normal plus cat and dog to an rgb triple.
 **********************************************************************/
void illuminate(double nn[3], double cat, double dog, GLfloat  rgb[3])
{
  double lmb, spec;
  /* illiLight by Ray Idaszak 1989 uses max{amb*lmb, rgb*lmb, spec}   */ 
  lmb = DOT(nn,luxx); lmb =(lmb<0 ? .2 : lmb); lmb = MAX(amb, lmb); 
  spec = CLAMP((1.1 - pwr+pwr*lmb), 0., 1.);  
  /* illiPaint by Chris Hartman 1993 maps R2(cat,dog)->R3(r,g,b)      */ 
  rgb[0] =  MAX(spec, lmb*dog);
  rgb[1] =  MAX(spec, lmb*(.25 + ABS(cat -.5)));
  rgb[2] =  MAX(spec, lmb*(1 - cat));
}

  private:
  double amb, pwr;
  double lux[3];	/* world light  unit (mostly)  vector    */
  double luxx[3];       /* object space  direction vector  */
};

IlliLuminator::IlliLuminator()
{
  double initialux[3]={1.0, 2.0, 3.0};
  initialize(0.3, 10.0, initialux);
}

void IlliLuminator::setLux(double newlux[3])
{
  LCPY(newlux, lux);

  // normalize light vector
  double magnitude=NRM(lux);
  for(int ii=0;ii<3;ii++)lux[ii] /= magnitude;  

  LCPY(lux, luxx);
}

void IlliLuminator::initialize(double amb, double pwr, double lux[3])
{
  this->amb = amb; 
  this->pwr = pwr;
  setLux(lux);
}

void IlliLuminator::increasePwr(double multiple)
{
  pwr*=multiple;
}

void IlliLuminator::increaseAmb(double multiple)
{
  amb*=multiple;
}

double IlliLuminator::getAmb(){return amb;}
double IlliLuminator::getPwr(){return pwr;}

IlliLuminator illuminator;

/**********************************************************************
 * IlliTorus class definition
 * The venerable torus, with its infamous cat-and-dogging.
 **********************************************************************/
class IlliTorus
{
  public:
  IlliTorus();
  void initialize();
  void draw();
  void increaseSmallSweep(int increment);
  void increaseGreatSweep(int increment);
  void grow();
  void shrink();
  void increaseGap(double multiple);
  double getGap();
  void setDefaultGap(double gapSize);

  private:
  void drawVertex(int th, int ta);

  //arrays are passed by reference, so this function puts the normal into nn
  void getNormal(int th, int ta, double nn[3])
  {
    /* radius of unit sphere is also unit normal to the torus           */
    nn[0] = C(th)*C(ta);
    nn[1] = S(th)*C(ta);
    nn[2] =       S(ta);
  }
  int th0, th1; //starting and ending meridians
  int ta0, ta1; //starting and ending latitudes
  int dth, dta;
  double gap;
  double gap0;
  enum{meridians=32};
  enum{latitudes=32};
  //static const int meridians=32;
  //static const int latitudes=32;
};

/**********************************************************************
 * IlliTorus method definitions
 **********************************************************************/
IlliTorus::IlliTorus(){initialize();}

void IlliTorus::initialize()
{
  //these numbers are angles in degrees
  th0=5; th1=355; 
  ta0=5; ta1=355;
  dth=(th1-th0)/meridians; dta=(ta1-ta0)/latitudes;

  //this is the space between meridian strips
  gap=gap0=1.0;
}

/**********************************************************************
 * small sweep is the arc of the small circle 
 * great sweep is the arc of the large circle
 * increment is applied to both ends of the arc
 **********************************************************************/
void IlliTorus::increaseSmallSweep(int increment)
{
  ta0-=increment;
  ta1+=increment;
  dta = (int)((ta1-ta0)/latitudes);  //increment in degrees
}

void IlliTorus::increaseGreatSweep(int increment)
{
  th0-=increment;
  th1+=increment;
  dth = (int)((th1-th0)/meridians);  //increment in degrees
}

void IlliTorus::grow()
{
  increaseSmallSweep(1); increaseGreatSweep(1);
}

void IlliTorus::shrink()
{
  increaseSmallSweep(-1); increaseGreatSweep(-1);
}

void IlliTorus::increaseGap(double multiple) { gap*=multiple; }
void IlliTorus::setDefaultGap(double gapSize) { gap0=gapSize; }
double IlliTorus::getGap(){return gap;}

/**********************************************************************
 * IlliTorus::draw
 * The functionality of drawtor packaged in a member function:
 * This method draws a torus.
 **********************************************************************/
void IlliTorus::draw()
{
  //th for the big circle, ta for the small one.  both in degrees
  int th=0, ta=0;

  if(dth==0 || dta==0){return;}

  for(th=th0; th < th1; th += dth){
   glBegin(GL_TRIANGLE_STRIP);
     for(ta = ta0 ; ta < ta1 ; ta += dta ){
         drawVertex(th,ta); drawVertex(int(th+gap*dth),ta);}
   glEnd();
  }/* end for.theta loop */
}

/**********************************************************************
 * IlliTorus::drawVertex
 * drawvert repackaged as a member function.  
 * Lights and draws a vertex of the torus
 **********************************************************************/
void IlliTorus::drawVertex(int th, int ta)
{
  double  nn[3], dog, cat;
  GLfloat  rgb[3];

  getNormal(th, ta, nn);	//normal goes in nn

  //dog and cat are for IlliPaint
  dog = (ta-ta0)/(double)(ta1-ta0); cat = (th-th0)/(double)(th1-th0);

  //put color into rgb
  illuminator.illuminate(nn, cat, dog, rgb);

  //set color
  glColor3fv(rgb);

  /* torus has unit small diameter and unit big radius                */
  glVertex3f( C(th) + .5*nn[0],
              S(th) + .5*nn[1],
                0   + .5*nn[2]);
} /* end drawvert */
IlliTorus torus;

/**********************************************************************
 * class AutoThymer
 * The savory (sorry, I couldn't resist) 
 * alternative to autotymer.  Each autothymer object is
 * a program for an animation.  The advance method tells it to 
 * move itself one step further in the animation.
 * marionette is the object whose strings will be pulled.
 **********************************************************************/
template <class TT>
class AutoThymer
{
  public:
  AutoThymer(TT& initMarionette);
  void addStep(void (TT::*fptr)(), int repetitions);
  void advance();
  void reset();

  private:
  TT& marionette;
  vector<pair<void (TT:: * )(), int> > commands;
  void (TT::*fptr)();
  int count;
  int max;
  int currentCommand;
};

template <class TT>
void AutoThymer<TT>::advance()
{
  if(commands.size() == 0) { return; }

  if(count >= max)
  {
    //switch to next command in sequence
    currentCommand++;
    currentCommand %= commands.size();
    fptr=commands[currentCommand].first;
    max=commands[currentCommand].second;
    count=0;
  }

  if(fptr)
  { 
    //execute command if it exists, otherwise rest
    (marionette.*fptr)();
  }
  count++;

}

template <class TT>
AutoThymer<TT>::AutoThymer(TT& marionette) : 
  marionette(marionette),
  count(0),
//  max(0),
  currentCommand(0),
  fptr(0)
{max=0;}

/**********************************************************************
 * addStep adds a step to the sequence.  
 * if you want a pause,
 * pass it 0 instead of a function
 **********************************************************************/
template <class TT>
void AutoThymer<TT>::addStep(void (TT::*fptr)(), int repetitions)
{
  commands.push_back(pair<void (TT::*)(), int>(fptr, repetitions));
  //whenever a step is added, the animation is reset
  count=0;
  this->fptr = commands[0].first;
  max=commands[0].second;
}

template <class TT>
void AutoThymer<TT>::reset()
{
  count=0;
  max=0;
  currentCommand=0;
  fptr=0;
}

AutoThymer<IlliTorus> thymer(torus);

/**********************************************************************/
void deFault(void){         /* (Z)ap also restores these assignments  */ 
  torus.initialize();
  selectF=0;
  selectG=0;
  initialF();
  initialG();
  computeAll();
  compute3DScale();
  msg=1; binoc=0; nose=.06; mode=TURNMODE;       /* gadget parameters */   
  speed=.02; torq=.006; focal = 2.; wfar=1000; mysiz=.01; morph=0; zoom=2.1;
  creep = 0; /* default 1 = no creep */
  for(int ii=0; ii<16; ii++) 
    {slatemat[ii]=starmat[ii]=aff[ii] = (ii/4==ii%4);}   /* identity matrix */ 
  double initialux[3]={1.0,2.0,3.0};		 /* light vector */
  illuminator.initialize(0.3, 10.0, initialux);  /* lighting params   */
  aff[12]=0; aff[13]= 0; aff[14]= -4.2; /* place where we can see it  */
  slatemat[14]=-5 ; initpoints(); dragmode=0; /* for BFarmer's picker */
  
  thymer.reset();
  thymer.addStep(&(IlliTorus::shrink), 150);
  thymer.addStep(0, 20);	//pause for a 20 count
  thymer.addStep(&(IlliTorus::grow), 150);
  thymer.addStep(0, 30);	//pause for a 30 count

  pickedcube[0] = pickedcube[1] = -1;  /* which cube is picked */
  /* Here are the defaults for the illiAnalyst keyboard commands */
  friz=1;
  drawcontrol=0;
  drawf=1;
  drawg=1;
  drawfhat=0;
  drawghat=0;
  drawfg=1;
  drawfghat=0;
  drawfhatghat=0; 
  drawcoord=1;
  linewidth=2;
  stars=1;
  scaleconvolution=1;
  background=1;
  drawcoordtopright=1;

}


/**********************************************************************/
void drawall(void){ 
	drawTransforms(); 
	if(drawcoord==1){
		drawCoordAxis();
	}
}
/**********************************************************************/
#if 0
void drawstars(void){  /* replace with SLevy's much prettier stars    */ 
  int ii,jj;
  double tmp;
  static GLfloat star[MANYSTARS][3]; static int virgin=1;
  if(virgin){             /* first time through set up the stars      */
     FOR(ii,0,MANYSTARS){ /* in a unit cube then  on unit sphere      */
       FOR(jj,0,3)star[ii][jj]  =(double)random()/RAND_MAX - 0.5;
       tmp=NRM(star[ii]); FOR(jj,0,3)star[ii][jj]/=tmp;        
     }
   virgin=0; /* never again */
  }
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();           /* optional insurance or superstition    */ 
    glMultMatrixf(starmat);
    glColor3f(0.8,0.9,1.0);        
    glBegin(GL_POINTS); 
      FOR(ii,0,MANYSTARS)glVertex3fv(star[ii]); 
    glEnd();
  /* glutWireTeapot(1);  if you prefer one on the firmament instead  */ 
  glPopMatrix();            /* optional insurance or superstition    */
  glClear(GL_DEPTH_BUFFER_BIT); /* put the stars at infinity         */ 
}
#endif
/**********************************************************************/
/*** SLevy's Stars -- mad props go out to the star pimp ***/
/**********************************************************************/
#include "opengl_stars.c"
/************************ steering ***********************************/
void arguments(int argc,char **argv){           /* Pat Hanrahan 1989 */
      while(--argc){++argv; if(argv[0][0]=='-')switch(argv[0][1]){
      case 'w': win =atoi(argv[1]); argv++; argc--; break; 
      case 'o': outfname = argv[1]; argv++; argc--; break;
      case 'g': torus.setDefaultGap(atof(argv[1])); argv++; argc--; break;
      case 'L': 
		double initialux[3];
		initialux[0]=atof(argv[1]);
                initialux[1]=atof(argv[2]);
                initialux[2]=atof(argv[3]); 
		illuminator.setLux(initialux);
		argv +=3; argc -=3; break;
   }}}
/**********************************************************************/
int number, hasnumber, decimal, sign;  /* globals for SLevy's gadgets */
   /* these are assigned in keyboard() but used by these factor fcns  */  
double getnumber(double dflt){    /*  from keyboard, factor of bump()   */ 
     double v = (sign ? -number : number);  /* positive or negative nr */
     if(!hasnumber) return dflt;       /* if no new nr use old on     */ 
     return decimal>0 ? v/(double)decimal : v;
}
void bump(double *val, double incr){          /* SLevy 98 */
  double by = fabs(incr);   /* wizard speak ... best not mess with it  */ 
  char fmt[8], code[32];
  int digits = 1;
  if(hasnumber) {
    *val = getnumber(0);
    return;
  }
  if(by <= .003) digits = 3;
  else if(by <= .03) digits = 2;
  sprintf(fmt, "%%.%de", digits);
  sprintf(code, fmt, *val * (1 + incr));
  sscanf(code, "%lf", val);
}
/********************from SLevy 2jan02 ********************************/
int getbutton(char key) { 
    int uu = clefs[key & 127]; clefs[key & 127]=0; return uu;
}
/**********************************************************************/
void keyboard(unsigned char key, int x, int y){
   clefs[key&127]=1;  /* globalize the keys that were pressed         */ 
#define  IF(K)            if(key==K)  
#define  PRESS(K,A,b)     IF(K){b;} IF(K-32){A;}  /* catch upper case */
#define  TOGGLE(K,flg)    IF(K){(flg) = 1-(flg); }
#define  CYCLE(K,f,m)   PRESS((K),(f)=(((f)+(m)-1)%(m)),(f)=(++(f)%(m)))
#define  SLIDI(K,f,m,M)   PRESS(K,(--f<m?m:f), (++f>M?M:f)) 
#define  SLIDF(K,f,m,M,d) PRESS(K,((f -= d)<m?m:f), ((f += d)>M?M:f))
/* Only 127 ASCII chars are processed in this GLUT callback function  */ 
/* Use the specialkeybo function for the special keys                 */ 
   IF(27) { exit(0); }                           /* ESC exit          */
   TOGGLE('v',binoc);                            /* cross-eyed STEREO */
   TOGGLE(' ',mode);                             /* space key         */
   TOGGLE('h',morph);                            /* autotymer on/off  */
   // Specific Instructions for Analyst 
   TOGGLE('b',friz);								 /* freeze on/off  */
   TOGGLE('t',stars);								 /* stars on/off  */
   TOGGLE('1',drawf);								 /* draw funcF  */
   TOGGLE('2',drawg);								 /* draw funcG  */
   TOGGLE('3',drawfg);								 /* draw funcFandG  */
   TOGGLE('4',drawfhat);                             /* draw funcFhat  */
   TOGGLE('5',drawghat);                             /* draw funcGhat  */
   TOGGLE('6',drawfghat);                            /* draw funcFandGhat  */
   TOGGLE('7',drawfhatghat);                         /* draw funcFhatGhat  */
   TOGGLE('8',drawcontrol);                          /* draw control points  */
   TOGGLE('9',drawcoord);                            /* draw control points  */
   TOGGLE('0',drawcoordtopright);                            /* draw control points  */
   TOGGLE('c',scaleconvolution);                     /* draw control points  */
   TOGGLE('d',background);                     /* draw control points  */
   CYCLE('l',linewidth,4);							 /* line width 1,2,3,4 */						 
   if(key=='f'){									 /* select function f */
	   selectF = (selectF + 1)%numfunctions;
	   initialF();
	   computeAll();
	   compute3DScale();
       initpoints();
   }  
   if(key=='g'){									 /* select function g */
	   selectG = (selectG + 1)%numfunctions;
	   initialG();
	   computeAll();
	   compute3DScale();
       initpoints();
   }


   CYCLE('w',msg,3);           /* writing on/off/speedometer+bullseye */
   PRESS('s', bump(&speed,.02), bump(&speed,-.02));/* flying speed    */ 
   PRESS('q', bump(&torq, .02), bump(&torq, -.02)); /* turning speed  */
   PRESS('m', creep+=.01, creep-=.01);     /* convolution */
   PRESS('z', deFault(), deFault());             /* zap changes       */

   /*PRESS('n', nose -= .001 , nose += .001 );*/     /* for binoculars    */
   /*PRESS('o', focal *= 1.1 , focal /= 1.1)*/       /* telephoto         */
   // PRESS('i', mysiz /= 1.1, mysiz *= 1.1)        /* rescale the world */
   PRESS('i', zoom /= 1.1, zoom *= 1.1)             /* rescale the world */
   /*PRESS('p', wfar *= 1.01 , wfar   /= 1.01)*/   /* rear clipping plane */
   /*if(key=='g'){torus.increaseGap(0.9);}*/
   /*if(key=='G'){torus.increaseGap(1.0/0.9);}*/
   /*if(key=='a'){illuminator.increaseAmb(1.0/0.9);}*/
   /*if(key=='A'){illuminator.increaseAmb(0.9);}*/
   /*if(key=='r'){illuminator.increasePwr(0.9);}*/
   /*if(key=='R'){illuminator.increasePwr(1.0/0.9);}*/
//   PRESS('a',amb /= .9, amb *= .9);             /* ambient fraction   */
//   PRESS('r',pwr /= .9, pwr *= .9);             /* pseudo-spec power  */
/********** SLevy's parser creates the input decimal ******************/
   if(key >= '0' && key <= '9'){         /* if key is a digit numeral */
          hasnumber = 1; number = number*10+key-'0'; decimal *= 10; } 
   else if(key == '.') { decimal = 1; }  /* it's a decimal !          */
   else if(key == '-') { sign = -1; }    /* it's negative  !          */
   else { hasnumber = number = decimal = sign = 0;} /* erase mess     */
   glutPostRedisplay();                 /* in case window was resized */
}
/**********************************************************************/
void specialkeybo(int key, int x, int y){
  clefs[0]= key ;
  switch(key){    /*  HOME END PAGE_DOWN RIGHT F1 etc  see glut.h    */
  case GLUT_KEY_F1:  torus.increaseGreatSweep(-1); break;
  case GLUT_KEY_F2:  torus.increaseGreatSweep( 1); break;
  /* default: fprintf(stderr,"non-ASCII char [%d] pressed.\n", key); */
  }
  hasnumber=number=decimal=0; glutPostRedisplay();
}
/**********************************************************************/


float deltat(int measurenow) {
	static float lastdeltat;
    static float smoothrate = 0.5;	// Smaller smoothrates average over more samples.  Must be < 1!
	static float weight;
	static float sum;

	if(!measurenow) return weight>0 ? sum/weight : 0;

#ifdef WIN
	static struct _timeb lnow, lthen;
	_ftime(&lnow);
    lastdeltat = lthen.time==0 ? 0 :
			(lnow.time - lthen.time) +
				.001 * (lnow.millitm - lthen.millitm);
	lthen = lnow;
#else
	static struct timeval now, then;
	gettimeofday(&now, NULL);
	lastdeltat = then.tv_sec==0 ? 0 :
			(now.tv_sec - then.tv_sec) +
				.000001 * (now.tv_usec - then.tv_usec);
	then = now;
#endif
	weight = weight*smoothrate + 1;
	sum = sum*smoothrate + lastdeltat;
	return sum / weight;
}

float speedometer(void){                      /* this one is for win32*/ 
  static double rate; static int ii=0;
  static float netdt;
  netdt += deltat(0);
  if(++ii % 8 == 0 && netdt > 0){                 /* 8 times around measure time */
	   rate = 8/netdt;
	   netdt = 0;
  }
  return((float)rate);
}

/**********************************************************************/
void char2wall(double x,double y,double z, char buf[]){
     char *p; glRasterPos3f(x,y,z);
     for(p = buf;*p;p++) glutBitmapCharacter(GLUT_BITMAP_9_BY_15,*p);
}
/**********************************************************************/
void messages(void){char buf [256]; 
  /* console messages are done differently from cave */
#define  LABEL(x,y,W,u) {sprintf(buf,(W),(u));char2wall(x,y,0.,buf);}
  glMatrixMode(GL_PROJECTION); glPushMatrix();  /* new projection matrix */
  glLoadIdentity(); gluOrtho2D(0,3000,0,3000);  /* new 2D coordinates */
  glMatrixMode(GL_MODELVIEW); glPushMatrix(); glLoadIdentity();
      if(mode==TURNMODE) glColor3f(1.,0.,1.); else glColor3f(1.,1.,0.);
      LABEL(1500,1500,"%s","o"); /* place a bullseye dead center */
      LABEL(80,170,"%4.1f fps",speedometer());
      if(msg!=2){              /* bypass the graffiti */
      LABEL(80,2840,\
      "(ESC)ape (V)Binoc (MAUS2)Fore (BAR)%s Change (F) (G) (W)riting",
             mode?"TURNMODE":"FLYMODE");

      LABEL(80,2700,"(S)peed  %0.4f",speed);
      LABEL(80,2630,"tor(Q) %0.4f",torq);
      LABEL (80,2560,"(Z)ap %s","");
      LABEL(80,2490,"freeze(B)",friz);
	  LABEL(80,2410,"s(T)ars",stars);
      LABEL(80,2330,"(I)zoom %.2g",zoom);

      glColor3f(1.,1.,0.);
	  LABEL(10, 10,"Illustrated Analyst by Henry, Shuman, Woodruff, Farmer, Nachand & Francis, \
 U Illinois, 2002 (c) %s","19july02");

/*      
LABEL(10,10,"Built on illiSkel-2002 by Francis, Levy, Bourd, Hartman,\
& Chappell, U Illinois, 1995..2002 %s","14jan02");	  
*/

	  glColor3f(1.,1.,1.); 
	  LABEL(80,1800,"F (1)",drawf);
	  
	  glColor3f(1.,0.,1.);
	  LABEL(80,1720,"G (2)",drawg);

	  glColor3f(0.,0.5,1.);
	  LABEL(80,1640,"FG (3)",drawfg);
	  
	  glColor3f(1.,1.,1.);
	  LABEL(80,1560,"F^ (4)",drawfhat);
	  
	  glColor3f(1.,0.,1.);
	  LABEL(80,1480,"G^ (5)",drawghat);
	  
	  glColor3f(0.5,1.0,0.);
	  LABEL(80,1400,"(FG)^ (6)",drawfghat);
	  
	  glColor3f(1.0,0.5,0.);
	  LABEL(80,1320,"(F^)(G^) (7)",drawfhatghat);

	  glColor3f(1.,0.,1.);
	  LABEL(80,1240,"Pick (8)",drawcontrol);
	  LABEL(80,1160,"Middle Axes (9)",drawcoord);
	  LABEL(80,1080,"Top Right Axes (0)",drawcoordtopright);
	  LABEL(80,1000,"(L)ine Width",linewidth);
	  LABEL(80,920,"S(C)ale FG (G)",selectG);
	  LABEL(80,840,"Slide (M)",creep);

	  /*LABEL(80,760,"Backgroun(D)",background);*/
	  /*LABEL(80,2770,"(N)ose   %0.3f",nose);*/
      /*LABEL(80,2560,"near clipper %g", mysiz*focal);*/
      /*LABEL(80,2490,"f(O)cal factor %g",focal);*/
      /*LABEL(80,2350,"far cli(P)per= %.2g",wfar); */
      /*LABEL(80,2210,"(G)ap %.2lg",torus.getGap());*/
      /*LABEL(80,2140,"(A)mb %.2g",illuminator.getAmb());*/
      /*LABEL(80,2070,"pw(R) %.2g",illuminator.getPwr());*/
    } /* end bypass*/ 
    glPopMatrix();
    glMatrixMode(GL_PROJECTION); glPopMatrix();
} 
/************************ navigation **********************************/
void chaptrack(int paw,int xx,int yy,int shif){/* Glenn Chappell 1992 */
   int dx,dy; 
   //if(friz){return;}
   float dt = deltat(1);		// Measure clock once per display cycle
   float motion = dt * 20;		// pseudo-time-scale for motion scaling
   dx = (int)(xx -.5*xwide); dx = abs(dx)>5?dx:0;        /* 5 pixel latency  */
   dy = (int)(yy -.5*yhigh); dy = abs(dy)>5?dy:0; 
   glMatrixMode(GL_MODELVIEW); glPushMatrix(); glLoadIdentity();
   if(mode==TURNMODE) glTranslatef(aff[12],aff[13],aff[14]);
   if(friz==0){
   glRotatef(dx*torq*motion,0.,1.,0.); glRotatef(dy*torq*motion,1.,0.,0.);
   if(paw&(1<<GLUT_RIGHT_BUTTON ))glRotatef(shif?-10:-1,0.,0.,1.);
   if(paw&(1<<GLUT_LEFT_BUTTON  ))glRotatef(shif?10:1,0.,0.,1.);
   } 
   if(mode==FLYMODE){
      glPushMatrix();
      glMultMatrixf(starmat);
      glGetFloatv(GL_MODELVIEW_MATRIX,starmat);

      glPopMatrix();
   }

   float mspeed = motion * speed;
   if(paw&(1<<GLUT_MIDDLE_BUTTON))glTranslatef(0.,0.,shif?-mspeed:mspeed);
   if(clefs[0]==GLUT_KEY_UP){ glTranslatef(0., mspeed, 0.);clefs[0]=0;}
   if(clefs[0]==GLUT_KEY_DOWN){ glTranslatef(0., -mspeed,0.);clefs[0]=0;}
   if(clefs[0]==GLUT_KEY_LEFT){ glTranslatef(-mspeed,0.,0.);clefs[0]=0;}
   if(clefs[0]==GLUT_KEY_RIGHT){ glTranslatef(mspeed,0.,0.);clefs[0]=0;}
   if(clefs[0]==GLUT_KEY_PAGE_UP){ glTranslatef(0.,0., mspeed);clefs[0]=0;}
   if(clefs[0]==GLUT_KEY_PAGE_DOWN){ glTranslatef(0.,0.,-mspeed);clefs[0]=0;}
   if(mode==TURNMODE) glTranslatef(-aff[12],-aff[13],-aff[14]);
   glMultMatrixf(aff); 
   glGetFloatv(GL_MODELVIEW_MATRIX,aff);
   int ii=0, jj=0;
   illuminator.transformLux(aff);
   glPopMatrix();
}
/************************* scenery ************************************/
void reshaped(int xx, int yy){xwide=xx ; yhigh=yy;} /*win width,height*/
/**********************************************************************/
void drawcons(void){ double asp =(double)xwide/yhigh; /* aspect ratio   */
  glEnable(GL_DEPTH_TEST);                    /* enable z-buffer */
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); 
  if (background==1){
	  glClearColor(0,0,0,0);   /* base color, try (.1,.2,.3,0.)           */  
  }
  else{
	  glClearColor(1.0,1.0,1.0,0);   /* base color, try (.1,.2,.3,0.)           */  
  }
  glLineWidth(linewidth + 1.0);

  if(binoc) glViewport(0,yhigh/4,xwide/2,yhigh/2);
  glMatrixMode(GL_PROJECTION); glLoadIdentity();
  glFrustum(-mysiz*asp,mysiz*asp,-mysiz,mysiz,mysiz*focal,wfar); 

  
  if(stars == 1){
	  float affeye[16];
	  memcpy(affeye, aff, sizeof(aff));
	  affeye[4*3+0] = affeye[4*3+1] = affeye[4*3+2] = 0;
	  glMatrixMode(GL_MODELVIEW);
	  glLoadMatrixf(starmat); 
	  drawstars();
	  glMatrixMode( GL_PROJECTION );
  }


  /* keep this here so that the picking stuff continues working */
 if(binoc==0){  /* EXCEPTIONAL CASE */
  glPushMatrix(); glLoadIdentity();
  glOrtho(-zoom*asp+.5,zoom*asp+.5,-zoom,zoom,zoom,100); 
  glMatrixMode(GL_MODELVIEW); glLoadIdentity();
  //if(stars == 1){ drawstars(); }
  glMultMatrixf(aff);
  drawall();
  glMatrixMode(GL_PROJECTION); 
  glPopMatrix();
  }else{  /* binoc==1 */ 
      /* left eye */
      glViewport(0,yhigh/4,xwide/2,yhigh/2);
      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity();
      drawstars();
      glTranslatef(binoc*nose,0.0,0.0);
      glMultMatrixf(aff);
      drawall();
      /* right eye */ 
      glViewport(xwide/2,yhigh/4,xwide/2,yhigh/2); 
      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity();
      drawstars();
      glTranslatef(binoc*nose,0.0,0.0);
      glMultMatrixf(aff);
      drawall();
    }
  drawslate();
  if(drawfg == 1){
	drawConvolution();
  }
  glViewport(0,0,xwide,yhigh);
  if(msg) messages(); 
  glutSwapBuffers();
}
/************************** steering **********************************/
void idle(void){            /* do this when nothing else is happening */
//   if(morph) autotymer(0);                       /* advance autotymer */ 
   if(morph) thymer.advance();
   glutPostRedisplay();                          /* redraw the window */
   IFCLICK('=',chaptrack(PAW,XX,YY,SHIF);)       /* bypass navigation */
   glDisable(GL_DEPTH_TEST);                   /* bypass depth buffer */ 
   IFCLICK('-',glEnable(GL_DEPTH_TEST); )      /* bypass depth buffer */
}
/**********************************************************************/

/**********************************************************************/
void mousepushed(int but,int stat,int x,int y){
  if(stat==GLUT_DOWN) PAW |= (1<<but); /*key came down and called back*/
  else PAW &= (-1 ^ (1<<but));              /* on the wayup erase flag*/ 
  XX=x; YY=y;        /* position in window coordinates (pos integers) */ 
  SHIF=(glutGetModifiers()==GLUT_ACTIVE_SHIFT)?1:0;  /* shift down too*/

   if(stat==GLUT_DOWN){ /* if a button was pushed */
      if(pCubeId != -1){  /* if we have a valid point picked */
         dragmode = 1 - dragmode; /* switch the dragmode */
      }
   }
}
/**********************************************************************/
/**********************************************************************/
#if 0  //fix this someday
void mousepushed(int but,int stat,int x,int y){
  if(stat==GLUT_DOWN) PAW |= (1<<but); /*key came down and called back*/
  else PAW &= (-1 ^ (1<<but));              /* on the wayup erase flag*/ 
  XX=x; YY=y;        /* position in window coordinates (pos integers) */ 
  SHIF=(glutGetModifiers()==GLUT_ACTIVE_SHIFT)?1:0;  /* shift down too*/
}
#endif
/**********************************************************************/
void mousemoved(int x,int y){ XX=x; YY=y; }
/***************** one ring to rule the all ***************************/
int main(int argc, char **argv){  
   arguments(argc,argv);                      /* from the commandline */

   deFault();                         /* values of control parameters */
       glutInit(&argc, argv);  
       glutInitDisplayMode(GLUT_DOUBLE|GLUT_DEPTH);
       switch(win){ 
           case 0: break;                   /* manage your own window */
           case 1: glutInitWindowSize(640, 480);
                   glutInitWindowPosition(100,100); break;
           case 2: glutInitWindowPosition(0,0); break;
         }
       if(win==2) glutFullScreen();
       glutCreateWindow("<* illiSkel 2002 in C/OpenGL/GLUT *>");
       glutDisplayFunc(drawcons); 
               /* the following are optional for interactive control  */ 
       glutKeyboardFunc(keyboard);
       glutSpecialFunc(specialkeybo);
       glutMouseFunc(mousepushed);
       glutMotionFunc(mousemoved);       
       glutPassiveMotionFunc(mousemoved); 
                                      /*  beyond here all are needed  */
       glutReshapeFunc(reshaped);
       glutIdleFunc(idle);             
       glutMainLoop();
	return 0;
}
/**********************************************************************/
 
