/*
This code is under development in the UIMATH.grafiXlab.
The history of this code is:
Nathaniel Thurston etal, Geometry Center "Outside In" sources
Michael McGuffin extended these codes to OpenGL on Linux
Byron Persino, Math290, Fall2000, adapted it to Windows
G. Francis
10feb01
*/
/*
   The complete source code can be downloaded from
   http://www.csclub.uwaterloo.ca/~mjmcguff/eversion/
   This file was written by Michael McGuffin, 1998.
   And modified to work on Windows98 by Byron Persion
   For Math/CS290, Jan 2001. 
   And subsequently cleaned up by George Francis 
   Want: Slevy style parameter changes
         illiView style headsup display 
*/
#include <stdio.h>    /* sprintf(), printf(), fprintf(), getchar() */
#include <stdlib.h>
#include <math.h>
#include <string.h>   /* memcpy(), strcat() */
#include <windows.h>
#include <GL\glut.h>	
// McGuffin's code has a separate generateGeometry.cpp and .h 
// which Persino kindly included in a single file here so that
// compilation becomes more userfriendly
// I have marked the beginning and end of the insertion  
// byron #include "generateGeometry.h" a 45 line header file
typedef unsigned char Boolean;
typedef unsigned long Pixel;

const double PI = 3.14159265358979323846264338327950;
const double M_PI = 3.14159265358979323846264338327950;

const float objectRadius = 1.5;
const float objectDistanceFromCamera = 1.0;

const double speedOfAnimatedRotation = 0.01; // must be between 0.0 and 1.0

// ==========================================================================

// global variables

int windowHeight;  //byron
int windowWidth;   //byron
int buttonState;  //byron

unsigned int minOfWidthAndHeight() {
      return windowWidth < windowHeight ? windowWidth : windowHeight;
}


double deltaTime = 1.0/64;


// display parameters
int False = 0;
int True = 1;
const int defaultNumberOfLatitudinalPatchesPerHemisphere = 12;
const int defaultNumberOfLongitudinalPatchesPerStrip = 12;
const int defaultNumStrips = 8;
Boolean bBackFaceCulling = False;
Boolean bFrontFacing = True;
Boolean bShadingIsSmooth = True;
typedef enum {
   style_points = 0,
   style_polygons,
   style_checkered,
   style_bands,
   number_of_styles
} RenderingStyle;
RenderingStyle renderingStyle = style_polygons;
Boolean bHalfStrips = False;
Boolean bAnimatedEversion = False;
Boolean bAnimatedEversionBackwards = False;
Boolean bAnimatedRotation = False;
Boolean bDisplayText = True;
#ifdef TOGGLEABLE_WINDOW_DECORATIONS
Boolean bNoWindowDecorations = False;
#endif
double angleOfMouseDragForAnimatedRotation = 0.0; // radians
Pixel currentTextColour;

// ==========================================================================
// GL Points are points on the surface of an object
// to be rendered.  They have a location (x,y,z) and a
// normal vector (nx,ny,nz).  Axis conventions:
// x+ right, y+ up, z+ out of the screen

struct GLPoint {

    // Coordinates stored in arrays are convenient for passing into OpenGL calls
    float vertex[3],   // contains location (x,y,z)
          normal[3];   // contains normal vector (nx,ny,nz)
};

typedef GLPoint * GLPointPointer; 

// ----------------------------------------

void generateGeometry(
   GLPoint ** geometryMatrix,   // Must be an array of (1 + u_count) arrays of (1 + v_count) elements
   double time = 0.0,   // must be between 0.0 and 1.0
   int numStrips = 8,   // If this is too small, there will be pinches in the eversion.

   // The u parameter corresponds to latitude.
   // Parallels [90 degrees north, 90 degrees south] are mapped to [0.0, 2.0].
   //
   double u_min = 0.0,   // Set to 0.0 to start at north pole
   int u_count = 12,     // Recommended value: 12*(u_max-u_min)
   double u_max = 1.0,   // Set to 1.0 to stop at equator, or 2.0 to stop at south pole

   // The v parameter corresponds to longitude.
   // Meridians [0 degrees, 180 degrees west] are mapped to [0, numStrips].
   //
   double v_min = 0.0,
   int v_count = 12,     // Recommended value: 12*(v_max-v_min)
   double v_max = 1.0,


   double bendtime = -1.0,   // -1 means don't do bendtime at all

   double corrStart   = 0.00,   // start of corrugation
   double pushStart   = 0.10,   // start of push (poles are pushed through each other)
   double twistStart  = 0.23,   // start of twist (poles rotate in opposite directions)
   double unpushStart = 0.60,   // start of unpush (poles held fixed while corrugations pushed through center)
   double uncorrStart = 0.93    // start of uncorrugation
);

// McGuffin's generateGeometry.cpp starts here but looks like
// this is a 1026 line source file ending 
// Nat Thurston's code .... should be checked someday
/*
    This file is part of a program called sphereEversion.
    sphereEversion was created by Michael McGuffin.
    The code in this file was almost entirely taken
    (with slight adaptations) from the source code of
    a program called "evert", which was written by
    Nathaniel Thurston.
    evert's source code can be down loaded from
        http://www.geom.umn.edu/docs/outreach/oi/software.html

    I (Michael McGuffin) wish to thank Nathaniel Thurston,
    Silvio Levy, and the Geometry Center (University of Minnesota)
    for making evert's source code freely available to the public.
*/

// ----------------------------------------

class TwoJet D(const class ThreeJet x, int index);
class TwoJet {
  public: /* this is a hack, but needed for now */
  double f;
  double fu, fv;
  double fuv;

  TwoJet() {}
  TwoJet(double d, double du, double dv)
   { f = d; fu = du; fv = dv; fuv = 0; }
  TwoJet(double d, double du, double dv, double duv)
   { f = d; fu = du; fv = dv; fuv = duv; }
#if 0
  operator double() { return f; }
#endif
  operator<(double d) { return f < d; }
  operator>(double d) { return f > d; }
  operator<=(double d) { return f <= d; }
  operator>=(double d) { return f >= d; }
  double df_du() { return fu; }
  double df_dv() { return fv; }
  double d2f_dudv() { return fuv; }
  void operator +=(TwoJet x)
   { f += x.f; fu += x.fu; fv += x.fv; fuv += x.fuv; }
  void operator +=(double d)
   { f += d; }
  void operator *=(TwoJet x)
   {
     fuv = f*x.fuv + fu*x.fv + fv*x.fu + fuv*x.f;
     fu = f*x.fu + fu*x.f;
     fv = f*x.fv + fv*x.f;
     f *= x.f;
   }
  void operator *=(double d)
   { f *= d; fu *= d; fv *= d; fuv *= d; }
  void operator %=(double d)
   { f = fmod(f, d); if (f < 0) f += d; }
  void operator ^=(double n)
   {
    if (f > 0) {
     double x0 = pow(f, n);
     double x1 = n * x0/f;
     double x2 = (n-1)*x1/f;
     fuv = x1*fuv + x2*fu*fv;
     fu = x1*fu;
     fv = x1*fv;
     f = x0;
    }
   }
  void Annihilate(int index)
   { if (index == 0) fu = 0;
     else if (index == 1) fv = 0;
     fuv = 0;
   }
  void TakeSin() {
   *this *= 2*M_PI;
   double s = sin(f), c = cos(f);
   f = s; fu = fu*c; fv = fv*c; fuv = c*fuv - s*fu*fv;
  }
  void TakeCos() {
   *this *= 2*M_PI;
   double s = cos(f), c = -sin(f);
   f = s; fu = fu*c; fv = fv*c; fuv = c*fuv - s*fu*fv;
  }
  
  friend TwoJet operator+(const TwoJet x, const TwoJet y);
  friend TwoJet operator*(const TwoJet x, const TwoJet y);
  friend TwoJet operator+(const TwoJet x, double d);
  friend TwoJet operator*(const TwoJet x, double d);
  friend TwoJet Sin(const TwoJet x);
  friend TwoJet Cos(const TwoJet x);
  friend TwoJet operator^(const TwoJet x, double n);
  friend TwoJet Annihilate(const TwoJet x, int index);
  friend TwoJet Interpolate(const TwoJet v1, const TwoJet v2, const TwoJet weight);
  friend void printJet(const TwoJet);
  friend class TwoJet D(const class ThreeJet x, int index);
  friend class ThreeJet;
};

// ----------------------------------------

TwoJet operator+(const TwoJet x, const TwoJet y) {
  return TwoJet(x.f+y.f, x.fu+y.fu, x.fv+y.fv, x.fuv + y.fuv);
}

TwoJet operator*(const TwoJet x, const TwoJet y) {
  return TwoJet(
    x.f*y.f,
    x.f*y.fu + x.fu*y.f,
    x.f*y.fv + x.fv*y.f,
    x.f*y.fuv + x.fu*y.fv + x.fv*y.fu + x.fuv*y.f
  );
}

TwoJet operator+(const TwoJet x, double d) {
  return TwoJet( x.f + d, x.fu, x.fv, x.fuv);
}

TwoJet operator*(const TwoJet x, double d) {
  return TwoJet( d*x.f, d*x.fu, d*x.fv, d*x.fuv);
}

TwoJet Sin(const TwoJet x) {
  TwoJet t = x*(2*M_PI);
  double s = sin(t.f);
  double c = cos(t.f);
  return TwoJet(s, c*t.fu, c*t.fv, c*t.fuv - s*t.fu*t.fv);
}

TwoJet Cos(const TwoJet x) {
  TwoJet t = x*(2*M_PI);
  double s = cos(t.f);
  double c = -sin(t.f);
  return TwoJet(s, c*t.fu, c*t.fv, c*t.fuv - s*t.fu*t.fv);
}

TwoJet operator^(const TwoJet x, double n) {
  double x0 = pow(x.f, n);
  double x1 = (x.f == 0) ? 0 : n * x0/x.f;
  double x2 = (x.f == 0) ? 0 : (n-1)*x1/x.f;
  return TwoJet(x0, x1*x.fu, x1*x.fv, x1*x.fuv + x2*x.fu*x.fv);
}

TwoJet Annihilate(const TwoJet x, int index) {
  return TwoJet(x.f, index == 1 ? x.fu : 0, index == 0 ? x.fv : 0, 0);
}

TwoJet Interpolate(const TwoJet v1, const TwoJet v2, const TwoJet weight) {
  return (v1) * ((weight) * (-1) + 1) + v2*weight;
}

void printJet(const TwoJet v) {
 printf("%f (%f %f)\n",
  v.f,
  v.fu, v.fv
 );
}

// ----------------------------------------

class ThreeJet {
public: // hack
  double f;
private:
  double fu, fv;
  double fuu, fuv, fvv;
  double fuuv, fuvv;

  ThreeJet(double d, double du, double dv, double duu, double duv, double dvv,
   double duuv, double duvv)
   { f = d; fu = du; fv = dv; fuu = duu; fuv = duv; fvv = dvv;
     fuuv = duuv; fuvv = duvv; }
  public:
  ThreeJet() {}
  ThreeJet(double d, double du, double dv)
   { f = d; fu = du; fv = dv; fuu = fuv = fvv = fuuv = fuvv = 0;}
  operator TwoJet() { return TwoJet(f, fu, fv, fuv); }
#if 0
  operator double() { return f; }
#endif
  operator<(double d) { return f < d; }
  operator>(double d) { return f > d; }
  operator<=(double d) { return f <= d; }
  operator>=(double d) { return f >= d; }
  void operator %=(double d)
   { f = fmod(f, d); if (f < 0) f += d; }
  friend ThreeJet operator+(const ThreeJet x, const ThreeJet y);
  friend ThreeJet operator*(const ThreeJet x, const ThreeJet y);
  friend ThreeJet operator+(const ThreeJet x, double d);
  friend ThreeJet operator*(const ThreeJet x, double d);
  friend ThreeJet Sin(const ThreeJet x);
  friend ThreeJet Cos(const ThreeJet x);
  friend ThreeJet operator^(const ThreeJet x, double n);
  friend ThreeJet Annihilate(const ThreeJet x, int index);
  friend ThreeJet Interpolate(const ThreeJet v1, const ThreeJet v2, const ThreeJet weight);
  friend void printJet(const ThreeJet);
  friend class TwoJet D(const class ThreeJet x, int index);
};

// ----------------------------------------

ThreeJet operator+(const ThreeJet x, const ThreeJet y) {
  ThreeJet result;
  result.f = x.f + y.f;
  result.fu = x.fu + y.fu;
  result.fv = x.fv + y.fv;
  result.fuu = x.fuu + y.fuu;
  result.fuv = x.fuv + y.fuv;
  result.fvv = x.fvv + y.fvv;
  result.fuuv = x.fuuv + y.fuuv;
  result.fuvv = x.fuvv + y.fuvv;
  return result;
}

ThreeJet operator*(const ThreeJet x, const ThreeJet y) {
  ThreeJet result;
  result.f = x.f*y.f;
  result.fu = x.f*y.fu + x.fu*y.f;
  result.fv = x.f*y.fv + x.fv*y.f;
  result.fuu = x.f*y.fuu + 2*x.fu*y.fu + x.fuu*y.f;
  result.fuv = x.f*y.fuv + x.fu*y.fv + x.fv*y.fu + x.fuv*y.f;
  result.fvv = x.f*y.fvv + 2*x.fv*y.fv + x.fvv*y.f;
  result.fuuv = x.f*y.fuuv + 2*x.fu*y.fuv + x.fv*y.fuu
           + 2*x.fuv*y.fu + x.fuu*y.fv + x.fuuv*y.f;
  result.fuvv = x.f*y.fuvv + 2*x.fv*y.fuv + x.fu*y.fvv
           + 2*x.fuv*y.fv + x.fvv*y.fu + x.fuvv*y.f;
  return result;
}

ThreeJet operator+(const ThreeJet x, double d) {
  ThreeJet result;
  result = x;
  result.f += d;
  return result;
}

ThreeJet operator*(const ThreeJet x, double d) {
  ThreeJet result;
  result.f = d*x.f;
  result.fu = d*x.fu;
  result.fv = d*x.fv;
  result.fuu = d*x.fuu;
  result.fuv = d*x.fuv;
  result.fvv = d*x.fvv;
  result.fuuv = d*x.fuuv;
  result.fuvv = d*x.fuvv;
  return result;
}

ThreeJet Sin(const ThreeJet x) {
  ThreeJet result;
  ThreeJet t = x*(2*M_PI);
  double s = sin(t.f);
  double c = cos(t.f);
  result.f = s;
  result.fu = c*t.fu;
  result.fv = c*t.fv;
  result.fuu = c*t.fuu - s*t.fu*t.fu;
  result.fuv = c*t.fuv - s*t.fu*t.fv;
  result.fvv = c*t.fvv - s*t.fv*t.fv;
  result.fuuv = c*t.fuuv - s*(2*t.fu*t.fuv + t.fv*t.fuu) - c*t.fu*t.fu*t.fv;
  result.fuvv = c*t.fuvv - s*(2*t.fv*t.fuv + t.fu*t.fvv) - c*t.fu*t.fv*t.fv;
  return result;
}

ThreeJet Cos(const ThreeJet x) {
  ThreeJet result;
  ThreeJet t = x*(2*M_PI);
  double s = cos(t.f);
  double c = -sin(t.f);
  result.f = s;
  result.fu = c*t.fu;
  result.fv = c*t.fv;
  result.fuu = c*t.fuu - s*t.fu*t.fu;
  result.fuv = c*t.fuv - s*t.fu*t.fv;
  result.fvv = c*t.fvv - s*t.fv*t.fv;
  result.fuuv = c*t.fuuv - s*(2*t.fu*t.fuv + t.fv*t.fuu) - c*t.fu*t.fu*t.fv;
  result.fuvv = c*t.fuvv - s*(2*t.fv*t.fuv + t.fu*t.fvv) - c*t.fu*t.fv*t.fv;
  return result;
}

ThreeJet operator^(const ThreeJet x, double n) {
  double x0 = pow(x.f, n);
  double x1 = (x.f == 0) ? 0 : n * x0/x.f;
  double x2 = (x.f == 0) ? 0 : (n-1) * x1/x.f;
  double x3 = (x.f == 0) ? 0 : (n-2) * x2/x.f;
  ThreeJet result;
  result.f = x0;
  result.fu = x1*x.fu;
  result.fv = x1*x.fv;
  result.fuu = x1*x.fuu + x2*x.fu*x.fu;
  result.fuv = x1*x.fuv + x2*x.fu*x.fv;
  result.fvv = x1*x.fvv + x2*x.fv*x.fv;
  result.fuuv = x1*x.fuuv + x2*(2*x.fu*x.fuv + x.fv*x.fuu) + x3*x.fu*x.fu*x.fv;
  result.fuvv = x1*x.fuvv + x2*(2*x.fv*x.fuv + x.fu*x.fvv) + x3*x.fu*x.fv*x.fv;
  return result;
}

TwoJet D(const ThreeJet x, int index) {
  TwoJet result;
  if (index == 0) {
    result.f = x.fu;
    result.fu = x.fuu;
    result.fv = x.fuv;
    result.fuv = x.fuuv;
  } else if (index == 1) {
    result.f = x.fv;
    result.fu = x.fuv;
    result.fv = x.fvv;
    result.fuv = x.fuvv;
  } else {
    result.f = result.fu = result.fv =
    result.fuv = 0;
  }
  return result;
}

ThreeJet Annihilate(const ThreeJet x, int index) {
  ThreeJet result = ThreeJet(x.f,0,0);
  if (index == 0) {
    result.fv = x.fv;
    result.fvv = x.fvv;
  } else if (index == 1) {
    result.fu = x.fu;
    result.fuu = x.fuu;
  }
  return result;
}

ThreeJet Interpolate(const ThreeJet v1, const ThreeJet v2, const ThreeJet weight) {
  return (v1) * ((weight) * (-1) + 1) + v2*weight;
}

void printJet(const ThreeJet v) {
 printf("%f (%f %f)\n",
  v.f,
  v.fu, v.fv
 );
}

// ----------------------------------------

struct TwoJetVec {
  TwoJet x;
  TwoJet y;
  TwoJet z;
  TwoJetVec() {}
  TwoJetVec(TwoJet a, TwoJet b, TwoJet c) { x = a; y = b; z = c; }
};

TwoJetVec operator+(TwoJetVec v, TwoJetVec w);
TwoJetVec operator*(TwoJetVec v, TwoJet  a);
TwoJetVec operator*(TwoJetVec v, double a);
TwoJetVec AnnihilateVec(TwoJetVec v, int index);
TwoJetVec Cross(TwoJetVec v, TwoJetVec w);
TwoJet Dot(TwoJetVec v, TwoJetVec w);
TwoJetVec Normalize(TwoJetVec v);
TwoJetVec RotateZ(TwoJetVec v, TwoJet angle);
TwoJetVec RotateY(TwoJetVec v, TwoJet angle);
TwoJetVec RotateX(TwoJetVec v, TwoJet angle);
TwoJetVec InterpolateVec(TwoJetVec v1, TwoJetVec v2, TwoJet weight);
TwoJet Length(TwoJetVec v);

// ----------------------------------------

TwoJetVec operator+(TwoJetVec v, TwoJetVec w) {
  TwoJetVec result;
  result.x = v.x + w.x;
  result.y = v.y + w.y;
  result.z = v.z + w.z;
  return result;
}

TwoJetVec operator*(TwoJetVec v, TwoJet  a) {
  TwoJetVec result;
  result.x = v.x*a;
  result.y = v.y*a;
  result.z = v.z*a;
  return result;
}

TwoJetVec operator*(TwoJetVec v, double a) {
  TwoJetVec result;
  result.x = v.x*a;
  result.y = v.y*a;
  result.z = v.z*a;
  return result;
}

TwoJetVec AnnihilateVec(TwoJetVec v, int index) {
  TwoJetVec result;
  result.x = Annihilate(v.x, index);
  result.y = Annihilate(v.y, index);
  result.z = Annihilate(v.z, index);
  return result;
}

TwoJetVec Cross(TwoJetVec v, TwoJetVec w) {
  TwoJetVec result;
  result.x = v.y*w.z + v.z*w.y*-1;
  result.y = v.z*w.x + v.x*w.z*-1;
  result.z = v.x*w.y + v.y*w.x*-1;
  return result;
}

TwoJet Dot(TwoJetVec v, TwoJetVec w) {
  return v.x*w.x + v.y*w.y + v.z*w.z;
}

TwoJetVec Normalize(TwoJetVec v) {
  TwoJet a;
  a = Dot(v,v);
  if (a > 0)
    a = a^-0.5;
  else
    a = TwoJet(0, 0, 0);
  return v*a;
}

TwoJetVec RotateZ(TwoJetVec v, TwoJet angle) {
  TwoJetVec result;
  TwoJet s,c;
  s = Sin (angle);
  c = Cos (angle);
  result.x =          v.x*c + v.y*s;
  result.y = v.x*s*-1 + v.y*c;
  result.z = v.z;
  return result;
}

TwoJetVec RotateY(TwoJetVec v, TwoJet angle) {
  TwoJetVec result;
  TwoJet s, c;
  s = Sin (angle);
  c = Cos (angle);
  result.x = v.x*c + v.z*s*-1;
  result.y = v.y;
  result.z = v.x*s + v.z*c    ;
  return result;
}

TwoJetVec RotateX(TwoJetVec v, TwoJet angle) {
  TwoJetVec result;
  TwoJet s,c;
  s = Sin (angle);
  c = Cos (angle);
  result.x = v.x;
  result.y = v.y*c + v.z*s;
  result.z = v.y*s*-1 + v.z*c;
  return result;
}

TwoJetVec InterpolateVec(TwoJetVec v1, TwoJetVec v2, TwoJet weight) {
  return (v1) * (weight*-1 + 1) + v2*weight;
}

TwoJet Length(TwoJetVec v)
{
  return (TwoJet(v.x^2) + TwoJet(v.y^2)) ^ (.5);
}

// ----------------------------------------

struct ThreeJetVec {
  ThreeJet x;
  ThreeJet y;
  ThreeJet z;
  operator TwoJetVec() { return TwoJetVec(x,y,z); }
};

ThreeJetVec operator+(ThreeJetVec v, ThreeJetVec w);
ThreeJetVec operator*(ThreeJetVec v, ThreeJet  a);
ThreeJetVec operator*(ThreeJetVec v, double a);
ThreeJetVec AnnihilateVec(ThreeJetVec v, int index);
ThreeJetVec Cross(ThreeJetVec v, ThreeJetVec w);
ThreeJet Dot(ThreeJetVec v, ThreeJetVec w);
TwoJetVec D(ThreeJetVec x, int index);
ThreeJetVec Normalize(ThreeJetVec v);
ThreeJetVec RotateZ(ThreeJetVec v, ThreeJet angle);
ThreeJetVec RotateY(ThreeJetVec v, ThreeJet angle);
ThreeJetVec RotateX(ThreeJetVec v, ThreeJet angle);
ThreeJetVec InterpolateVec(ThreeJetVec v1, ThreeJetVec v2, ThreeJet weight);
ThreeJet Length(ThreeJetVec v);

// ----------------------------------------

ThreeJetVec operator+(ThreeJetVec v, ThreeJetVec w) {
  ThreeJetVec result;
  result.x = v.x + w.x;
  result.y = v.y + w.y;
  result.z = v.z + w.z;
  return result;
}

ThreeJetVec operator*(ThreeJetVec v, ThreeJet  a) {
  ThreeJetVec result;
  result.x = v.x*a;
  result.y = v.y*a;
  result.z = v.z*a;
  return result;
}

ThreeJetVec operator*(ThreeJetVec v, double a) {
  ThreeJetVec result;
  result.x = v.x*a;
  result.y = v.y*a;
  result.z = v.z*a;
  return result;
}

ThreeJetVec AnnihilateVec(ThreeJetVec v, int index) {
  ThreeJetVec result;
  result.x = Annihilate(v.x, index);
  result.y = Annihilate(v.y, index);
  result.z = Annihilate(v.z, index);
  return result;
}

TwoJetVec D(ThreeJetVec x, int index) {
  TwoJetVec result;
  result.x = D(x.x, index);
  result.y = D(x.y, index);
  result.z = D(x.z, index);
  return result;
}

ThreeJetVec Cross(ThreeJetVec v, ThreeJetVec w) {
  ThreeJetVec result;
  result.x = v.y*w.z + v.z*w.y*-1;
  result.y = v.z*w.x + v.x*w.z*-1;
  result.z = v.x*w.y + v.y*w.x*-1;
  return result;
}

ThreeJet Dot(ThreeJetVec v, ThreeJetVec w) {
  return v.x*w.x + v.y*w.y + v.z*w.z;
}

ThreeJetVec Normalize(ThreeJetVec v) {
  ThreeJet a;
  a = Dot(v,v);
  if (a > 0)
    a = a^-0.5;
  else
    a = ThreeJet(0, 0, 0);
  return v*a;
}

ThreeJetVec RotateZ(ThreeJetVec v, ThreeJet angle) {
  ThreeJetVec result;
  ThreeJet s,c;
  s = Sin (angle);
  c = Cos (angle);
  result.x =          v.x*c + v.y*s;
  result.y = v.x*s*-1 + v.y*c;
  result.z = v.z;
  return result;
}

ThreeJetVec RotateY(ThreeJetVec v, ThreeJet angle) {
  ThreeJetVec result;
  ThreeJet s, c;
  s = Sin (angle);
  c = Cos (angle);
  result.x = v.x*c + v.z*s*-1;
  result.y = v.y;
  result.z = v.x*s + v.z*c    ;
  return result;
}

ThreeJetVec RotateX(ThreeJetVec v, ThreeJet angle) {
  ThreeJetVec result;
  ThreeJet s,c;
  s = Sin (angle);
  c = Cos (angle);
  result.x = v.x;
  result.y = v.y*c + v.z*s;
  result.z = v.y*s*-1 + v.z*c;
  return result;
}

ThreeJetVec InterpolateVec(ThreeJetVec v1, ThreeJetVec v2, ThreeJet weight) {
  return (v1) * (weight*-1 + 1) + v2*weight;
}

ThreeJet Length(ThreeJetVec v)
{
  return (ThreeJet(v.x^2) + ThreeJet(v.y^2)) ^ (.5);
}

// ----------------------------------------

TwoJetVec FigureEight(TwoJetVec w, TwoJetVec h, TwoJetVec bend, TwoJet form, TwoJet v) {

   TwoJet height;
   v %= 1;
   height = (Cos (v*2) + -1) * (-1);
   if (v > 0.25 && v < 0.75)
      height = height*-1 + 4;
   height = height*0.6;
   h = h + bend*(height*height*(1/64.0));
   return w*Sin (v*2) + (h) * (Interpolate((Cos (v) + -1) * (-2), height, form)) ;
}

TwoJetVec AddFigureEight(ThreeJetVec p, ThreeJet u, TwoJet v, ThreeJet form, ThreeJet scale, int numStrips) {

   ThreeJet size = form * scale;
   form = form*2 + form*form*-1;
   TwoJetVec dv = AnnihilateVec(D(p, 1), 1);
   p = AnnihilateVec(p, 1);
   TwoJetVec du = Normalize(D(p, 0));
   TwoJetVec h = Normalize(Cross(du, dv))*TwoJet(size);
   TwoJetVec w = Normalize(Cross(h, du))*(TwoJet(size)*1.1);
   return RotateZ(
      TwoJetVec(p) +
      FigureEight(w, h, du*D(size, 0)*(D(u, 0)^(-1)), form, v),
      v*(1.0/numStrips)
   );
}

// ----------------------------------------

ThreeJetVec Arc(ThreeJet u, ThreeJet v, double xsize, double ysize, double zsize) {

   ThreeJetVec result;
   u = u*0.25;
   result.x = Sin (u) * Sin (v) * xsize;
   result.y = Sin (u) * Cos (v) * ysize;
   result.z = Cos (u) * zsize;
   return result;
}

ThreeJetVec Straight(ThreeJet u, ThreeJet v, double xsize, double ysize, double zsize) {

   ThreeJetVec result;
   u = u*0.25;
#if 0
   u = (u) * (-0.15915494) + 1; /* 1/2pi */
#endif
   result.x = Sin (v) * xsize;
   result.y = Cos (v) * ysize;
   result.z = Cos (u) * zsize;
   return result;
}

ThreeJet Param1(ThreeJet x) {

   double offset = 0;
   x %= 4;
   if (x > 2) { x = x+(-2); offset = 2; }
   if (x <= 1) return x*2 + (x^2)*(-1) + offset;
   else return (x^2) + x*(-2) + (2 + offset);
}

ThreeJet Param2(ThreeJet x) {

   double offset = 0;
   x %= 4;
   if (x > 2) { x = x+(-2); offset = 2; }
   if (x <= 1) return (x^2) + offset;
   else return (x^2)*(-1) + x*4 + (-2 + offset);
}

static inline ThreeJet TInterp(double x) {
   return ThreeJet(x,0,0);
}

ThreeJet UInterp(ThreeJet x) {

   x %= 2;
   if (x > 1)
      x = x*(-1) + 2;
   return (x^2)*3 + (x^3) * (-2);
}

#define FFPOW 3
ThreeJet FFInterp(ThreeJet x) {

   x %= 2;
   if (x > 1)
      x = x*(-1) + 2;
   x = x*1.06 + -0.05;
   if (x < 0) return ThreeJet(0, 0, 0);
   else if (x > 1) return ThreeJet(0, 0, 0) + 1;
   else return (x ^ (FFPOW-1)) * (FFPOW) + (x^FFPOW) * (-FFPOW+1);
}

#define FSPOW 3
ThreeJet FSInterp(ThreeJet x) {

   x %= 2;
   if (x > 1)
      x = x*(-1) + 2;
   return ((x ^ (FSPOW-1)) * (FSPOW) + (x^FSPOW) * (-FSPOW+1)) * (-0.2);
}

ThreeJetVec Stage0(ThreeJet u, ThreeJet v) {
   return Straight(u, v, 1, 1, 1);
}

ThreeJetVec Stage1(ThreeJet u, ThreeJet v) {
   return Arc(u, v, 1, 1, 1);
}

ThreeJetVec Stage2(ThreeJet u, ThreeJet v) {
   return InterpolateVec(
      Arc(Param1(u), v, 0.9, 0.9, -1),
      Arc(Param2(u), v, 1, 1, 0.5), 
      UInterp(u)
   );
}

ThreeJetVec Stage3(ThreeJet u, ThreeJet v) {

   return InterpolateVec(
      Arc(Param1(u), v,-0.9,-0.9,-1), 
      Arc(Param2(u), v,-1, 1,-0.5),
      UInterp(u)
   );
}

ThreeJetVec Stage4(ThreeJet u, ThreeJet v) {
   return Arc(u, v, -1,-1, -1);
}

ThreeJetVec Scene01(ThreeJet u, ThreeJet v, double t) {
   return InterpolateVec(Stage0(u,v), Stage1(u,v), TInterp(t));
}

ThreeJetVec Scene12(ThreeJet u, ThreeJet v, double t) {
   return InterpolateVec(Stage1(u,v), Stage2(u,v), TInterp(t));
}

ThreeJetVec Scene23(ThreeJet u, ThreeJet v, double t) {

   ThreeJet tmp = TInterp(t);
   t = tmp.f * 0.5;
   double tt = (u <= 1) ? t : -t;
   return InterpolateVec(
      RotateZ(Arc(Param1(u), v, 0.9, 0.9,-1), ThreeJet(tt,0,0)),
      RotateY(Arc(Param2(u), v, 1, 1, 0.5), ThreeJet(t,0,0)),
      UInterp(u)
  );
}

ThreeJetVec Scene34(ThreeJet u, ThreeJet v, double t) {
   return InterpolateVec(Stage3(u,v), Stage4(u,v), TInterp(t));
}

TwoJetVec BendIn(ThreeJet u, ThreeJet v, double t, int numStrips) {

   ThreeJet tmp = TInterp(t);
   t = tmp.f;
   return AddFigureEight(
      Scene01(u, ThreeJet(0, 0, 1), t),
      u, v, ThreeJet(0, 0, 0), FSInterp(u),
      numStrips
   );
}

TwoJetVec Corrugate(ThreeJet u, ThreeJet v, double t, int numStrips) {

   ThreeJet tmp = TInterp(t);
   t = tmp.f;
   return AddFigureEight(
      Stage1(u, ThreeJet(0, 0, 1)),
      u, v, FFInterp(u) * ThreeJet(t,0,0), FSInterp(u),
      numStrips
   );
}

TwoJetVec PushThrough(ThreeJet u, ThreeJet v, double t, int numStrips) {

   return AddFigureEight(
      Scene12(u,ThreeJet(0, 0, 1),t),
      u, v, FFInterp(u), FSInterp(u),
      numStrips
   );
}

TwoJetVec Twist(ThreeJet u, ThreeJet v, double t, int numStrips) {

   return AddFigureEight(
      Scene23(u,ThreeJet(0, 0, 1),t),
      u, v, FFInterp(u), FSInterp(u),
      numStrips
   );
}

TwoJetVec UnPush(ThreeJet u, ThreeJet v, double t, int numStrips) {

   return AddFigureEight(
      Scene34(u,ThreeJet(0, 0, 1),t),
      u, v, FFInterp(u), FSInterp(u),
      numStrips
   );
}

TwoJetVec UnCorrugate(ThreeJet u, ThreeJet v, double t, int numStrips) {

   ThreeJet tmp;
   tmp = TInterp((t) * (-1) + 1);
   t = tmp.f;

   return AddFigureEight(
      Stage4(u,ThreeJet(0, 0, 1)),
      u, v, FFInterp(u) * ThreeJet(t,0,0), FSInterp(u),
      numStrips
   );
}

// ----------------------------------------

#ifdef SPLINE /* Please refer to comments in the Makefile about this #define */
void print_point (TwoJetVec p, double ps, double pus, double pvs, double puvs)
{
    float xyz[3];
    xyz[0] = (p.x.f)*ps + p.x.df_du()*pus/3.0 + p.x.df_dv()*pvs/3.0
             + p.x.d2f_dudv()*puvs/9.0;
    xyz[1] = (p.y.f)*ps + p.y.df_du()*pus/3.0 + p.y.df_dv()*pvs/3.0
             + p.y.d2f_dudv()*puvs/9.0;
    xyz[2] = (p.z.f)*ps + p.z.df_du()*pus/3.0 + p.z.df_dv()*pvs/3.0
             + p.z.d2f_dudv()*puvs/9.0;
    printf("%g %g %g\n", xyz[0], xyz[1], xyz[2]);
}
void printSpline(
   TwoJetVec v00, TwoJetVec v01,
   TwoJetVec v10, TwoJetVec v11,
   double us, double vs,
   double s0, double s1, double t0, double t1
) {
    print_point(v00, 1, 0, 0, 0);
    print_point(v00, 1, us, 0, 0);
    print_point(v10, 1,-us, 0, 0);
    print_point(v10, 1, 0, 0, 0);

    print_point(v00, 1, 0, vs, 0);
    print_point(v00, 1, us, vs, us*vs);
    print_point(v10, 1,-us, vs,-us*vs);
    print_point(v10, 1, 0, vs, 0);

    print_point(v01, 1, 0,-vs, 0);
    print_point(v01, 1, us,-vs,-us*vs);
    print_point(v11, 1,-us,-vs, us*vs);
    print_point(v11, 1, 0,-vs, 0);

    print_point(v01, 1, 0, 0, 0);
    print_point(v01, 1, us, 0, 0);
    print_point(v11, 1,-us, 0, 0);
    print_point(v11, 1, 0, 0, 0);

    printf("%g %g  %g %g  %g %g  %g %g\n\n", s0,t0,  s1,t0,  s0,t1, s1,t1);
}
#endif

void printMesh(TwoJetVec p, GLPoint * point) {

    double x = p.x.f ;
    double y = p.y.f ;
    double z = p.z.f ;
    double nx = p.y.df_du()*p.z.df_dv()-p.z.df_du()*p.y.df_dv();
    double ny = p.z.df_du()*p.x.df_dv()-p.x.df_du()*p.z.df_dv();
    double nz = p.x.df_du()*p.y.df_dv()-p.y.df_du()*p.x.df_dv();
    double s = nx*nx + ny*ny + nz*nz;
    if (s > 0) s = sqrt(1/s);

    /* printf("%f %f %f    %f %f %f\n", x, y, z, nx*s, ny*s, nz*s); */

    point->vertex[0] = x;
    point->vertex[1] = y;
    point->vertex[2] = z;
    point->normal[0] = -nx*s;
    point->normal[1] = -ny*s;
    point->normal[2] = -nz*s;
}

// ----------------------------------------

typedef TwoJetVec SurfaceTimeFunction(ThreeJet u, ThreeJet v, double t, int numStrips);

static inline double sqr(double x) {
  return x*x;
}
static inline double calcSpeedV(TwoJetVec v) {
  return sqrt(sqr(v.x.df_dv()) + sqr(v.y.df_dv()) + sqr(v.z.df_dv()));
}
static inline double calcSpeedU(TwoJetVec v) {
  return sqrt(sqr(v.x.df_du()) + sqr(v.y.df_du()) + sqr(v.z.df_du()));
}

void printScene(
   SurfaceTimeFunction *func,
   double umin, double umax, int ucount,
   double vmin, double vmax, int vcount,
   double t,
   GLPoint ** geometryMatrix,
   int numStrips
) {
   static TwoJetVec **values;
   int j, k;
   double u, v, delta_u, delta_v;

   if (ucount <= 0 || vcount <= 0) return;
   delta_u = (umax-umin) / ucount;
   delta_v = (vmax-vmin) / vcount;
   values = (TwoJetVec **) calloc(ucount+1, sizeof(TwoJetVec *));
   double *speedv = (double *) calloc(ucount+1, sizeof(double));
   double **speedu = (double **) calloc(ucount+1, sizeof(double *));
   for (j = 0; j <= ucount; j++) {
      u = umin + j*delta_u;
      values[j] = (TwoJetVec *) calloc(vcount+1, sizeof(TwoJetVec));
      speedu[j] = (double *) calloc(vcount+1, sizeof(double));
      speedv[j] = calcSpeedV((*func)(ThreeJet(u, 1, 0), ThreeJet(0, 0, 1), t, numStrips));
      if (speedv[j] == 0) {
         /* Perturb a bit, hoping to avoid degeneracy */
         u += (u < 1) ? 1e-9 : -1e-9;
         speedv[j] = calcSpeedV((*func)(ThreeJet(u, 1, 0), ThreeJet(0, 0, 1), t, numStrips));
      }
      for (k = 0; k <= vcount; k++) {
         v = vmin + k*delta_v;
         values[j][k] = (*func)( ThreeJet(u, 1, 0), ThreeJet(v, 0, 1), t, numStrips );
         speedu[j][k] = calcSpeedU(values[j][k]);
      }
   }

#ifdef SPLINE /* Please refer to comments in the Makefile about this #define */

   /* bezier code */

   for (j = 0; j < ucount; ++j) {
      u = umin + j*delta_u;
      for (k = 0; k < vcount; ++k) {
         v = vmin + k*delta_v;
         printSpline(
            values[j][k], values[j][k+1],
            values[j+1][k], values[j+1][k+1],
            delta_u, delta_v,
            umin + j*delta_u, umin + (j+1)*delta_u,  vmin + k*delta_v, vmin + (k+1)*delta_v
         );
      }
   }

#else

   /* quadrilateral mesh code */

   for (j = 0; j <= ucount; ++j)
      for (k = 0; k <= vcount; ++k) {
         printMesh(values[j][k],  &geometryMatrix[j][k]);
      }

#endif

   /* clean up */
   for (j = 0; j <= ucount; j++) {
      free(values[j]);
      free(speedu[j]);
   }
   free(values);
   free(speedu);
   free(speedv);
}

// ----------------------------------------

/*
   Refer to generateGeometry.h for
   documentation on this function.
*/
void generateGeometry(
   GLPoint ** geometryMatrix,
   double time,
   int numStrips,

   double u_min,
   int u_count,
   double u_max,
   double v_min,
   int v_count,
   double v_max,

   double bendtime,

   double corrStart,
   double pushStart,
   double twistStart,
   double unpushStart,
   double uncorrStart
) {
#ifndef SPLINE /* Please refer to comments in the Makefile about this #define */
   if (NULL == geometryMatrix)
      return;
#endif

   if (bendtime >= 0.0) {
      printScene(BendIn, u_min, u_max, u_count, v_min, v_max, v_count, bendtime, geometryMatrix, numStrips );
   } else {

      /* time = (time - howfar) / chunk */

      if (time >= uncorrStart)
         printScene(UnCorrugate, u_min, u_max, u_count, v_min, v_max, v_count,
		   (time - uncorrStart) / (1.0 - uncorrStart), geometryMatrix, numStrips );
      else if (time >= unpushStart)
         printScene(UnPush, u_min, u_max, u_count, v_min, v_max, v_count,
		   (time - unpushStart) / (uncorrStart - unpushStart), geometryMatrix, numStrips );
      else if (time >= twistStart)
         printScene(Twist, u_min, u_max, u_count, v_min, v_max, v_count,
		   (time - twistStart) / (unpushStart - twistStart), geometryMatrix, numStrips );
      else if (time >= pushStart)
         printScene(PushThrough, u_min, u_max, u_count, v_min, v_max, v_count,
		   (time - pushStart) / (twistStart - pushStart), geometryMatrix, numStrips );
      else if (time >= corrStart)
         printScene(Corrugate, u_min, u_max, u_count, v_min, v_max, v_count,
		   (time - corrStart) / (pushStart - corrStart), geometryMatrix, numStrips );
   }
}

#ifdef SPLINE /* Please refer to comments in the Makefile about this #define */
main() {
   generateGeometry(NULL);
}
#endif
//
// end of generateGeometry.cc by McGuffin/Thurston
//

void FreePixelColour(Pixel p) {

//byron    XFreeColors(XStuff.display, XStuff.colourMap, &p, 1, 0);
}

// ==========================================================================

// ==============================================================================

// The below bitmaps were created with the command "bitmap [filename]".

#define cursor_width 25
#define cursor_height 25
#define cursor_x_hot 12
#define cursor_y_hot 12
static char cursor_bits[] = {
   0xff, 0xff, 0xff, 0xff, 0xdf, 0xff, 0xf7, 0xff, 0xdf, 0xff, 0xf7, 0xff,
   0xaf, 0xff, 0xeb, 0xff, 0xaf, 0xff, 0xeb, 0xff, 0x77, 0xff, 0xdd, 0xff,
   0x77, 0xff, 0xdd, 0xff, 0xfb, 0xfe, 0xbe, 0xff, 0xfb, 0xfe, 0xbe, 0xff,
   0x01, 0x7c, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
   0xff, 0xd7, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xef, 0xff, 0xff,
   0xff, 0xd7, 0xff, 0xff, 0xff, 0xd7, 0xff, 0xff, 0xff, 0xbb, 0xff, 0xff,
   0xff, 0xbb, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff, 0xff, 0x7d, 0xff, 0xff,
   0xff, 0x00, 0xfe, 0xff};

#define mask_width 25
#define mask_height 25
#define mask_x_hot 12
#define mask_y_hot 12
static char mask_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x08, 0x00, 0x20, 0x00, 0x08, 0x00,
   0x70, 0x00, 0x1c, 0x00, 0x70, 0x00, 0x1c, 0x00, 0xf8, 0x00, 0x3e, 0x00,
   0xf8, 0x00, 0x3e, 0x00, 0xfc, 0x01, 0x7f, 0x00, 0xfc, 0x01, 0x7f, 0x00,
   0xfe, 0x83, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
   0x00, 0x38, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
   0x00, 0x38, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00,
   0x00, 0x7c, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00,
   0x00, 0xff, 0x01, 0x00};


// ==============================================================================

class Frustum {
public:
   void Init();
   void WindowResize();
   void Magnify(int delta);  // delta can be positive (to zoom in) or negative (to zoom out)
   void Reset();

   double Width() { return width; }
   double Height() { return height; }
   double LeftEdge() { return center_x - width/2; }
   double TopEdge() { return center_y + height/2; }
   double NearPlane() { return nearP; }
private:
   double unMagnifiedSize;
   double size;
   double width;   // width of nearP plane
   double height;  // height of nearP plane
   double nearP;    // distance to nearP plane from observer
   double farP;     // distance to farP plane from observer
   double center_x;
   double center_y;

   // magnificationFactor == magIncrement ^ magExponent
   //
   double magnificationFactor;  // 1.0 by default, larger values shrink the (width,height) proportionally
   int magExponent;             // 0 by default
   static const double magIncrement;
};

const double Frustum::magIncrement = 1.05;

void Frustum::Init() {

   nearP = objectDistanceFromCamera;
   farP = nearP + 4*objectRadius;
   center_x = 0.0;
   center_y = 0.0;

   magnificationFactor = 1.0;
   magExponent = 0;

   // Create clipping volume.
   // Let r be objectRadius, d be objectDistanceFromCamera.
   // We have two similar right triangles
   //
   //                         |
   //                  |?     |r
   // eye  ____________|______|
   //            d        r
   //
   //
   //  ? = alpha = dr/(r+d)
   //
   // But this value of alpha makes the object unpleasantly
   // large, so we multiply by a constant greater than 1.0
   //
   double alpha = 1.25 * objectDistanceFromCamera * objectRadius / (objectRadius + objectDistanceFromCamera);
   unMagnifiedSize = 2*alpha;
   size = unMagnifiedSize / magnificationFactor;
   WindowResize();
////   printf("Init called WindowResize\n");
}

void Frustum::WindowResize() {
////printf("WindowResize called\n");
    glViewport(0,0,windowWidth,windowHeight);

	if (windowWidth < windowHeight) {
			width = size;
			height = size * windowHeight/windowWidth;
	}
		else {
			height = size;
			width = size * windowWidth/windowHeight;
			}

////printf("Height=%d, Width=%d\n",windowHeight,windowWidth);


   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();   // clear matrix
   
   // for orthographic projection, use glOrtho() instead of glFrustum()
   //
    glFrustum(
       center_x - width/2, center_x + width/2, /* left,right */
      center_y - height/2, center_y + height/2, /* bottom, top */
      nearP, farP /* nearP, farP */
   );
 }




void Frustum::Magnify(int delta) {

   int j;

   magExponent += delta;
   magnificationFactor = 1.0;
   if (magExponent < 0)
      for (j = magExponent; j < 0; ++j)
         magnificationFactor /= magIncrement;
   else for (j = 0; j < magExponent; ++j)
      magnificationFactor *= magIncrement;

   size = unMagnifiedSize / magnificationFactor;
   WindowResize();
////   printf("Magnify called WindowResize\n");
}

void Frustum::Reset() {

   center_x = 0;
   center_y = 0;
   magExponent = 0;
   magnificationFactor = 1.0;
   size = unMagnifiedSize / magnificationFactor;
   WindowResize();
////   printf("Reset called WindowResize\n");
}

// ==============================================================================

Frustum frustum;

void changeSize(int w, int h) {
	windowHeight = h;
    windowWidth = w;
////printf("ChangeSize called\n");

	// Prevent a divide by zero, when window is too short
	// (you cant make a window of zero width).
	if(h == 0)
		h = 1;
//	if ( w < h) {
//		h = frustum.size * h/w;
//		else {
//	    w = frustum.size * w/h;

	float ratio = 1.0* w / h;

	// Reset the coordinate system before modifying
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	
	// Set the viewport to be the entire window
	glViewport(0, 0, w, h);

	// Set the correct perspective.
	gluPerspective(45,ratio,1,1000);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(0.0,0.0,5.0, 
		      0.0,0.0,-1.0,
			  0.0f,1.0f,0.0f);


}


// ==============================================================================

class Sphere {
public:
    Sphere(double time = 0,int numStrips = 0,int numHemispheresToDisplay = 0,int numStripsToDisplay = 0,int numberOfLatitudinalPatchesPerHemisphere = 0,int numberOfLongitudinalPatchesPerStrip = 0) :
        matrixOfVertices(NULL) { Construct(time,numStrips,numHemispheresToDisplay,numStripsToDisplay,numberOfLatitudinalPatchesPerHemisphere,numberOfLongitudinalPatchesPerStrip); }
    ~Sphere();
    void Construct(double time = 0,int numStrips = 0,int numHemispheresToDisplay = 0,int numStripsToDisplay = 0,int numberOfLatitudinalPatchesPerHemisphere = 0,int numberOfLongitudinalPatchesPerStrip = 0);
    void Reconstruct() { DeallocateMatrix(); GenerateVertices(); }
    void Draw();
    void IncrementTime(double deltaTime) { if (Time < 1.0) { DeallocateMatrix(); Time+=deltaTime; if (deltaTime > 1.0) deltaTime = 1.0; GenerateVertices(); } }
    void DecrementTime(double deltaTime) { if (Time > 0.0) { DeallocateMatrix(); Time-=deltaTime; if (deltaTime < 0.0) deltaTime = 0.0; GenerateVertices(); } }
    void IncrementNumStrips() { DeallocateMatrix(); ++NumStrips; ++NumStripsToDisplay; GenerateVertices(); }
    void DecrementNumStrips() { if (NumStrips > 1) { DeallocateMatrix(); --NumStrips; if (NumStripsToDisplay > 1) --NumStripsToDisplay; GenerateVertices(); } }
    void ChangeNumHemispheresToDisplay() { if (NumHemispheresToDisplay == NumHemispheres) NumHemispheresToDisplay = 1; else ++NumHemispheresToDisplay; }
    void IncrementNumStripsToDisplay() { if (NumStripsToDisplay < NumStrips) {++NumStripsToDisplay;} }
    void DecrementNumStripsToDisplay() { if (NumStripsToDisplay > 1) {--NumStripsToDisplay;} }
    void IncrementLatitudinalResolution() { DeallocateMatrix(); ++NumberOfLatitudinalPatchesPerHemisphere; GenerateVertices(); }
    void DecrementLatitudinalResolution() { DeallocateMatrix(); --NumberOfLatitudinalPatchesPerHemisphere; GenerateVertices(); }
    void IncrementLongitudinalResolution() { DeallocateMatrix(); ++NumberOfLongitudinalPatchesPerStrip; GenerateVertices(); }
    void DecrementLongitudinalResolution() { DeallocateMatrix(); --NumberOfLongitudinalPatchesPerStrip; GenerateVertices(); }

    double GetTime() { return Time; }

private:
    void GenerateVertices();
    void DeallocateMatrix();

    double Time;   // between 0.0 and 1.0

    static const int NumHemispheres;
    int NumStrips;
    int NumHemispheresToDisplay;
    int NumStripsToDisplay;
    int NumberOfLatitudinalPatchesPerHemisphere,
        NumberOfLongitudinalPatchesPerStrip;

    // This matrix stores all the vertices and normals used to render the
    // sphere.  Elements in the matrix are arranged by [latitude][longitude].
    //
    GLPoint ** matrixOfVertices;
};

const int Sphere::NumHemispheres = 2;

Sphere::~Sphere() {

    DeallocateMatrix();
}

void Sphere::DeallocateMatrix() {

    int j;
    
    if (matrixOfVertices == NULL)
       return;
    for (j = NumberOfLatitudinalPatchesPerHemisphere; j >= 0; --j)
       delete [] (matrixOfVertices[j]);
    delete [] matrixOfVertices;
    matrixOfVertices = NULL;
}

void Sphere::Construct(
    double time,
    int numStrips,
    int numHemispheresToDisplay,
    int numStripsToDisplay,
    int numberOfLatitudinalPatchesPerHemisphere,
    int numberOfLongitudinalPatchesPerStrip
) {
    if (time < 0.0)
       Time = 0.0;
    else if (time > 1.0)
       Time = 1.0;
    else Time = time;

    NumStrips = numStrips < 1 ? defaultNumStrips : numStrips;

    NumHemispheresToDisplay = numHemispheresToDisplay < 1 ? NumHemispheres : numHemispheresToDisplay;

    NumStripsToDisplay = numStripsToDisplay < 1 ? NumStrips : numStripsToDisplay;

    if (numberOfLatitudinalPatchesPerHemisphere <= 0)
       NumberOfLatitudinalPatchesPerHemisphere = defaultNumberOfLatitudinalPatchesPerHemisphere;
    else
       NumberOfLatitudinalPatchesPerHemisphere = numberOfLatitudinalPatchesPerHemisphere;

    if (numberOfLongitudinalPatchesPerStrip <= 0)
       NumberOfLongitudinalPatchesPerStrip = defaultNumberOfLongitudinalPatchesPerStrip;
    else
       NumberOfLongitudinalPatchesPerStrip = numberOfLongitudinalPatchesPerStrip;

    GenerateVertices();
}

void Sphere::GenerateVertices() {

    int j;

    // clamp input parameters to their minima
    if (Time < 0.0)
       Time = 0.0;
    else if (Time > 1.0)
       Time = 1.0;

    if (NumberOfLatitudinalPatchesPerHemisphere < 2)
       NumberOfLatitudinalPatchesPerHemisphere = 2;

    if (NumberOfLongitudinalPatchesPerStrip < 2)
       NumberOfLongitudinalPatchesPerStrip = 2;

    // throw away previous geometry
    DeallocateMatrix();

    // allocate stuff
    matrixOfVertices = new GLPointPointer[1 + NumberOfLatitudinalPatchesPerHemisphere];
    for (j = NumberOfLatitudinalPatchesPerHemisphere; j >= 0; --j)
       matrixOfVertices[j] = new GLPoint[1 + NumberOfLongitudinalPatchesPerStrip];

    // generate the geometry
    generateGeometry(
       matrixOfVertices,
       Time,
       NumStrips,

       0.0,
       NumberOfLatitudinalPatchesPerHemisphere,
       1.0,
       0.0,
       NumberOfLongitudinalPatchesPerStrip,
       bHalfStrips ? 0.5 : 1.0
#ifdef BEND_IN /* this will display a cylindar bending into a sphere */
       ,Time
#endif
    );

}

void Sphere::Draw() {

    // See Chapter 2 in the "OpenGL Programming Guide" (IRIS InSight online books)
    // for more info on triangle strips.
    // In a GL_TRIANGLE_STRIP block, if the vertices sent in are numbered 0,1,2,3,...
    // the resulting strip will look like
    //
    //     0---2---4---6---8--- ...
    //      \ / \ / \ / \ / \
    //       1---3---5---7---9 ...
    //
    // and will be cowposed of triangles (0,1,2), (2,1,3), (2,3,4), ...
    // The ordering is important.  By convention, polygons whose vertices appear in
    // counterclockwise order on the screen are front-facing.  If you create polygons
    // with vertices in the wrong order, and back-face culling is turned on, you'll
    // never see the polygons appear on the screen.

    int hemisphere,strip,j,k;

    glFrontFace(GL_CW);  // we're going to use the opposite convention

    glMatrixMode(GL_MODELVIEW);

    for (hemisphere = 0; hemisphere < NumHemispheresToDisplay; ++hemisphere) {
        glPushMatrix();
        glRotatef(hemisphere*180.0,0.0,1.0,0.0);
        for (strip = 0; strip < NumStripsToDisplay; ++strip) {
            glPushMatrix();
            glRotatef((hemisphere == 0 ? -strip : strip+1)*360.0/NumStrips,0.0,0.0,1.0);

            if (renderingStyle == style_points) {
                glBegin(GL_POINTS);
                for (j = 0; j <= NumberOfLatitudinalPatchesPerHemisphere; ++j)
                    for (k = 0; k <= NumberOfLongitudinalPatchesPerStrip; ++k) {
                        glNormal3fv(matrixOfVertices[j][k].normal);
                        glVertex3fv(matrixOfVertices[j][k].vertex);
                    }
                glEnd();
            }
            else {
                for (j = 0; j < NumberOfLatitudinalPatchesPerHemisphere; ++j) {
                    if (renderingStyle == style_polygons || (renderingStyle == style_bands && (j & 1)==hemisphere)) {
                        glBegin(GL_TRIANGLE_STRIP);
                        for (k = 0; k <= NumberOfLongitudinalPatchesPerStrip; ++k) {
                            glNormal3fv(matrixOfVertices[j][k].normal);
                            glVertex3fv(matrixOfVertices[j][k].vertex);
                            glNormal3fv(matrixOfVertices[j+1][k].normal);
                            glVertex3fv(matrixOfVertices[j+1][k].vertex);
                        }
                        glEnd();
                    }
                    else if (renderingStyle == style_checkered) {
                        for (k = j%2; k < NumberOfLongitudinalPatchesPerStrip; k+=2) {
                            glBegin(GL_TRIANGLE_STRIP);
                            glNormal3fv(matrixOfVertices[j][k].normal);
                            glVertex3fv(matrixOfVertices[j][k].vertex);
                            glNormal3fv(matrixOfVertices[j+1][k].normal);
                            glVertex3fv(matrixOfVertices[j+1][k].vertex);
                            glNormal3fv(matrixOfVertices[j][k+1].normal);
                            glVertex3fv(matrixOfVertices[j][k+1].vertex);
                            glNormal3fv(matrixOfVertices[j+1][k+1].normal);
                            glVertex3fv(matrixOfVertices[j+1][k+1].vertex);
                            glEnd();
                        }
                    }
                }
            }
            glPopMatrix();
        }
        glPopMatrix();
    }
}

// ==============================================================================

Sphere sphere;

// ==============================================================================

class RotationInformation {
public:
    RotationInformation()
        : centerX(0), centerY(0), viewportRadius(10),
          isClutchOn(False),
          previousDeviceX(0), previousDeviceY(0),
          rotationMode(arcball),
          elevation(0.0), azimuth(0.0)
    {
        CopyIdentityInto(oldRotationMatrix);
        CopyIdentityInto(rotationMatrix); 
    }

    void CalibrateDevice(int center_x, int center_y, int viewport_radius) {
        centerX = center_x;
        centerY = center_y;
        viewportRadius = viewport_radius;
    }

    Boolean IsClutchOn() { return isClutchOn; }
    void TurnClutchOn(int deviceX, int deviceY) {    // mouse button down
        isClutchOn = True;
        previousDeviceX = deviceX;
        previousDeviceY = deviceY;
    }
    void TurnClutchOff() {   // mouse button up
        isClutchOn = False;
    }
    void DeviceMotion(float deviceX, float deviceY);   // mouse motion

    enum RotationMode {   // CycleRotationMode() assumes that the first mode equals zero
        elevationAzimuth = 0,
        aroundXYAxes,
        virtualSphere,
        arcball,
        numberOfRotationModes
    };
    RotationMode GetRotationMode() { return rotationMode; }
    void SetRotationMode(RotationMode newMode);
    void CycleRotationMode() { SetRotationMode((RotationMode)((rotationMode+1) % (numberOfRotationModes))); }

    float * GetRotationMatrix() { return rotationMatrix; }
private:
    void CopyIdentityInto(float * matrix);

    int centerX, centerY, viewportRadius;
    Boolean isClutchOn;
    int previousDeviceX, previousDeviceY;
    RotationMode rotationMode;
    float oldRotationMatrix[16];   // cumulative past rotations
    float rotationMatrix[16];   // oldRotationMatrix + current rotation

    // stuff specific to "elevationAzimuth" mode
    float elevation, azimuth;   // in degrees; not radians
};






void RotationInformation::DeviceMotion(float deviceX, float deviceY) {

    // Here, the device axis conventions are X+ right, Y+ down
    // Note: glRotatef(theta, x, y, z); rotates theta degrees counter clockwise around vector (x,y,z)

    // The device was dragged deltaX pixels to the right and deltaY pixels down.
    float deltaX = deviceX - previousDeviceX,
          deltaY = deviceY - previousDeviceY;
    float pixelsPerDegree = 0.01*viewportRadius;

////    printf("DeviceMotion -- deltax=%d deltay=%d\n",deltaX,deltaY);
    if (!isClutchOn)
       return;
   

    glMatrixMode(GL_MODELVIEW);   // set transformation mode to model/view transformations
    glLoadIdentity();             // clear matrix

    switch (rotationMode) {
    case elevationAzimuth:
//		printf("elevationAzimuth\n");
        elevation += deltaY/pixelsPerDegree;
        azimuth += deltaX/pixelsPerDegree;
        glRotatef(elevation,1.0,0.0,0.0);   // perform rotation around x+ axis (pointing right)
        glRotatef(azimuth,0.0,1.0,0.0);   // perform rotation around y+ axis (pointing up)
        glMultMatrixf(oldRotationMatrix);   // preserve previous rotations
        // don't save into old
        break;
    case aroundXYAxes:
//gkf	printf("aroundXYAxes\n");
        glRotatef(deltaY/pixelsPerDegree,1.0,0.0,0.0);   // perform rotation around x+ axis (pointing right)
        glRotatef(deltaX/pixelsPerDegree,0.0,1.0,0.0);   // perform rotation around y+ axis (pointing up)
        glMultMatrixf(oldRotationMatrix);   // preserve previous rotations
        glGetFloatv(GL_MODELVIEW_MATRIX, oldRotationMatrix);   // save result into old
        break;
    case virtualSphere:
//gkf		printf("virtualSphere\n");
    case arcball:

        float ux, uy, uz,   // vector u corresponding to previous mouse position, where |u| = 1
              uzSquared,
              vx, vy, vz,   // vector v corresponding to new mouse position, where |v| = 1
              wx, wy, wz,   // vector w is cross product (u x v)
              vzSquared;
        float theta,        // angle (in radians) between u and v
              dotProduct,
              shrinkFactor;

        // compute u
        //
        ux = (previousDeviceX - centerX) / (float)viewportRadius;
        uy = (centerY - previousDeviceY) / (float)viewportRadius;   // uy+ points up, not down
        uzSquared = 1.0 - ux*ux - uy*uy;
        if (uzSquared < 0.0) {
           // Shorten vector to force a magnitude of one
           shrinkFactor = sqrt(ux*ux + uy*uy);
           ux /= shrinkFactor;
           uy /= shrinkFactor;
           uz = 0.0;
        }
        else uz = sqrt(uzSquared);

        // compute v
        //
        vx = (deviceX - centerX) / (float)viewportRadius;
        vy = (centerY - deviceY) / (float)viewportRadius;   // vy+ points up, not down
        vzSquared = 1.0 - vx*vx - vy*vy;
        if (vzSquared < 0.0) {
           // Shorten vector to force a magnitude of one
           shrinkFactor = sqrt(vx*vx + vy*vy);
           vx /= shrinkFactor;
           vy /= shrinkFactor;
           vz = 0.0;
        }
        else vz = sqrt(vzSquared);

        // compute cross product w = (u x v)
        //
        wx =   uy*vz - vy*uz;
        wy = -(ux*vz - vx*uz);
        wz =   ux*vy - vx*uy;

        // Since
        //    |w| = |u||v||sin theta|
        // and
        //    |u| = |v| = 1
        // we have
        //    |w| = |sin theta|
        // But theta = asin(|w|) only if 0 <= theta <= pi/2 radians.
        // If theta > pi/2, theta != asin(|w|).
        //
        // So we calculate theta using the dot product (u.v)
        //    u.v = |u||v|(cos theta) = cos theta
        //    theta = acos(u.v)
        // This works for 0 <= theta <= pi radians.
        dotProduct = ux*vx + uy*vy + uz*vz;
        if (dotProduct < -1.0)
            dotProduct = -1.0;
        else if (dotProduct > 1.0)
            dotProduct = 1.0;
        theta = acos(dotProduct);
        if (rotationMode == arcball) {
            // This is the only difference between virtualSphere and arcball.
            // Note that this number could be changed to achieve a generalization
            // of virtualSphere/arcball ... although virtualSphere is the only mode
            // which does not exhibit hysterisis.
            theta *= 2.0;
        }
        glRotatef(180 * theta / PI, wx, wy, wz);
////		printf(" wx =%d  wy=%d  wz=%d\n",wx,wy,wz);
        glMultMatrixf(oldRotationMatrix);   // preserve previous rotations
        glGetFloatv(GL_MODELVIEW_MATRIX, oldRotationMatrix);   // save result into old

        break;
    };
    
    glGetFloatv(GL_MODELVIEW_MATRIX, rotationMatrix);   // save result

    // Forgetting these two lines made for very interesting behaviour.
    // (Some bugs are cool !)  Try commenting them and see what happens ...
    previousDeviceX = deviceX;
    previousDeviceY = deviceY;
}

void RotationInformation::SetRotationMode(RotationMode newMode) {

    // Save rotations of current mode so they won't be lost,
    // allowing us to transfer smoothly into the new mode.
    memcpy(oldRotationMatrix,   // destination
           rotationMatrix,   // source
           16*sizeof(float));

    // reset mode-specific parameters
    elevation = 0.0;
    azimuth = 0.0;
    
    // set to new mode
    rotationMode = newMode;
}

void RotationInformation::CopyIdentityInto(float * matrix) {

    matrix[ 0] = 1.0;   matrix[ 1] = 0.0;   matrix[ 2] = 0.0;   matrix[ 3] = 0.0;
    matrix[ 4] = 0.0;   matrix[ 5] = 1.0;   matrix[ 6] = 0.0;   matrix[ 7] = 0.0;
    matrix[ 8] = 0.0;   matrix[ 9] = 0.0;   matrix[10] = 1.0;   matrix[11] = 0.0;
    matrix[12] = 0.0;   matrix[13] = 0.0;   matrix[14] = 0.0;   matrix[15] = 1.0;
}

// ==============================================================================

RotationInformation Rotator;


void setOrthographicProjection() {

	// switch to projection mode
	glMatrixMode(GL_PROJECTION);
	// save previous matrix which contains the 
	//settings for the perspective projection
	glPushMatrix();
	// reset matrix
	glLoadIdentity();
	// set a 2D orthographic projection
	gluOrtho2D(0, windowWidth, 0, windowHeight);
	// invert the y axis, down is positive
	glScalef(1, -1, 1);
	// mover the origin from the bottom left corner
	// to the upper left corner
	glTranslatef(0, -windowHeight, 0);
	glMatrixMode(GL_MODELVIEW);
}

int font=(int)GLUT_BITMAP_8_BY_13;
int font1=(int)GLUT_BITMAP_HELVETICA_18;


void renderBitmapString(float x, float y, void *font,char *string)
{
  
  char *c;
  glRasterPos2f(x, y);
  for (c=string; *c != '\0'; c++) {
    glutBitmapCharacter(font, *c);
  }
}

void resetPerspectiveProjection() {
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
}






// ==============================================================================

void Refresh() {										

   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);   // clear image & z-buffer

   glMatrixMode(GL_MODELVIEW);   // set transformation mode to model/view transformations
   glLoadIdentity();             // clear matrix

   glTranslatef(0.0, 0.0, -(objectRadius + objectDistanceFromCamera));   // move object away from origin, out in front of camera
   glMultMatrixf(Rotator.GetRotationMatrix());   // perform rotations

   sphere.Draw();
   glFlush();
   char buffer[40];

         strcpy(buffer,"(R)otation mode = ");
         switch(Rotator.GetRotationMode()) {
            case RotationInformation::elevationAzimuth : strcat(buffer,"elevation-azimuth"); break;
            case RotationInformation::aroundXYAxes     : strcat(buffer,"x-y");               break;
            case RotationInformation::virtualSphere    : strcat(buffer,"virtual sphere");    break;
            case RotationInformation::arcball          : strcat(buffer,"arcball");           break;
         }

   glPushMatrix();
	glColor3f(0.0f,1.0f,1.0f);
	setOrthographicProjection();
	glLoadIdentity();
	renderBitmapString(5,10,(void *)font, buffer);
	
	sprintf(buffer,"(H)omotopy = %.4f", sphere.GetTime());
    renderBitmapString(5,30,(void *)font1, buffer);

	sprintf(buffer,"d(T)ime = 1 / %d", (int)(1.0/deltaTime + 0.5));
	renderBitmapString(5,50,(void *)font1, buffer);

	resetPerspectiveProjection();
	glMatrixMode(GL_MODELVIEW);//byron
	glPopMatrix();
	glPopMatrix();
   
   glutSwapBuffers();


   
   

    if (bDisplayText) {

      // draw strings
      //
      
  //byron    int x_offset = 5, y_offset = 0, charHeight;

   ////   if (NULL != XStuff.smallFont) {
   ////      XSetFont(XStuff.display, XStuff.gc, XStuff.smallFont->fid);
   ////      charHeight = 1.2 * ( XStuff.smallFont->ascent + XStuff.smallFont->descent );

		 
////         XDrawString(XStuff.display, XStuff.window, XStuff.gc, x_offset, y_offset + charHeight, buffer, strlen(buffer));
////         y_offset += charHeight;
////      }

///Byron     if (NULL != XStuff.bigFont) {
///Byron	XSetFont(XStuff.display, XStuff.gc, XStuff.bigFont->fid);
///Byron         charHeight = 1.2 * ( XStuff.bigFont->ascent + XStuff.bigFont->descent );

         
///Byron         XDrawString(XStuff.display, XStuff.window, XStuff.gc, x_offset, y_offset + charHeight, buffer, strlen(buffer));
///Byron         y_offset += charHeight;
         
///Byron         XDrawString(XStuff.display, XStuff.window, XStuff.gc, x_offset, y_offset + charHeight, buffer, strlen(buffer));
      }

      // If the user hits the key to advance time many times in rapid
      // succession, the cpu will start doing some heavy computations
      // to try and keep up, displaying frame after frame as time
      // advances.  Meanwhile, requests to the X server
      // to draw strings can queue up, and finally all get dumped
      // onto the last frame.  The user sees this as text
      // overwriting itself, becoming illegible.  By calling
      // XFlush(), we prevent the requests from queuing up.
      //
///Byron      XFlush(XStuff.display);
   //}
}

  
// ==============================================================================

void MainLoop() {

   Boolean bQuit = False;
 
   
   while (!bQuit) {
      if (bAnimatedEversion || bAnimatedRotation) {
//byron         while (0 == XEventsQueued(XStuff.display, QueuedAfterFlush)) {
            // Perform animation.
            //
            if (bAnimatedEversion) {
               if (bAnimatedEversionBackwards) {
                  if (sphere.GetTime() == 0.0)
                     bAnimatedEversionBackwards = ! bAnimatedEversionBackwards;
                  else
                     sphere.DecrementTime(deltaTime);
               }
               else {
                  if (sphere.GetTime() == 1.0)
                     bAnimatedEversionBackwards = ! bAnimatedEversionBackwards;
                  else
                     sphere.IncrementTime(deltaTime);
               }
            }
            if (bAnimatedRotation) {
               // Do not rotate if the user is
               // currently trying to rotate.
               //
               if (!Rotator.IsClutchOn()) {
                  double lengthOfMouseDrag;
                  lengthOfMouseDrag = minOfWidthAndHeight()*0.5*speedOfAnimatedRotation;
                  Rotator.TurnClutchOn( windowWidth/2, windowHeight/2 );
                  Rotator.DeviceMotion(
                     windowWidth/2  + lengthOfMouseDrag * cos(angleOfMouseDragForAnimatedRotation),
                     windowHeight/2 + lengthOfMouseDrag * sin(angleOfMouseDragForAnimatedRotation)
                  );
                  Rotator.TurnClutchOff();
                  angleOfMouseDragForAnimatedRotation += 0.01;
                  if (angleOfMouseDragForAnimatedRotation > 2*PI)
                     angleOfMouseDragForAnimatedRotation -= 2*PI;
               }
            }
            Refresh();
         }
      }

//byron       XNextEvent(XStuff.display, &event);
//byron switch (event.type) {
//byron          case Expose:
//byron             // Unless this is the last contiguous expose, don't redraw.
//byron             if (event.xexpose.count != 0)
//byron                break;
//byron             // Only *contiguous* expose events can be skipped.  Searching through the event queue
//byron             // to find the last expose event would skip over intervening ConfigureNotify events.
//byron 
Refresh();
//byron             break;
//byron          case ConfigureNotify:
//byron             // window has been resized
//byron             if (
//byron                XStuff.windowWidth != event.xconfigure.width ||
//byron                XStuff.windowHeight != event.xconfigure.height
//byron             ) {
//byron                XStuff.windowWidth = event.xconfigure.width;
//byron                XStuff.windowHeight = event.xconfigure.height;

   ////            frustum.WindowResize();
   ////            Rotator.CalibrateDevice(XStuff.windowWidth/2, XStuff.windowHeight/2, XStuff.minOfWidthAndHeight()/2);
   ////            Refresh();
//byron             }
//byron             break;
//byron          case KeyPress:
 /*       }   /* switch(event */
   }   /* while (!bQuit) */








void processNormalKeys(unsigned char key, int x, int y) {

	if (key == 27) 
		exit(0);

	switch(key) {
//gkf better keys for homotopy 

       case '+': //XK_KP_Add:
               // case XK_plus:
                  sphere.IncrementTime(deltaTime);
                  Refresh();
                  break;
 case 'h': sphere.IncrementTime(deltaTime); Refresh(); break;
 case 'H': sphere.DecrementTime(deltaTime); Refresh(); break;
 case 'T': deltaTime = deltaTime <.5?2*deltaTime:1.0; Refresh(); break;
 case 't': if(deltaTime*0.5>=1.0/8192)deltaTime*=0.5; Refresh(); break; 
        case '-': //XK_KP_Subtract:
               // case XK_minus:
                  sphere.DecrementTime(deltaTime);
                  Refresh();
                  break;

        case '*': //XK_KP_Multiply:
                  deltaTime *= 2.0;
                  if (deltaTime > 1.0)
                     deltaTime = 1.0;
                  Refresh();
                  break;
        case '/': //XK_KP_Divide:
                  if (deltaTime * 0.5 >= 1.0/8192)
                     deltaTime *= 0.5;
                  Refresh();
                  break;


		case 0X20: //XK_space:
                  renderingStyle = (RenderingStyle)(((int)renderingStyle + 1) % number_of_styles);
                  Refresh();
                  break;
	case 'S': //XK_S:
	case 's': //XK_s:
                  bShadingIsSmooth = !bShadingIsSmooth;
                  glShadeModel(bShadingIsSmooth ? GL_SMOOTH : GL_FLAT);
                  Refresh();
                  break;
	case 'b': //XK_b:
	case 'B': //XK_B:
                  // toggle back face culling
                  bBackFaceCulling = !bBackFaceCulling;
                  if (bBackFaceCulling) {
                     glEnable(GL_CULL_FACE);
                     glCullFace(bFrontFacing ? GL_BACK : GL_FRONT);
                  }
                  else
                     glDisable(GL_CULL_FACE);
                  Refresh();
                  break;
	case 'f': // XK_f:
	case 'F': //XK_F:
                  bFrontFacing = !bFrontFacing;
                  glCullFace(bFrontFacing ? GL_BACK : GL_FRONT);
                  Refresh();
                  break;
	case 'r': //XK_r:
	case 'R': //XK_R:
                  // cycle forward to new rotation mode
                  Rotator.CycleRotationMode();
                  Refresh();
                  break;
/*	case '1': //XK_1:
	case '2': //XK_2:
	case '3': //XK_3:
	case '4': //XK_4:
	case '5': //XK_5:
	case '6': //XK_6:
	case '7': //XK_7:
	case '8': //XK_8:
                  // change colour of something
                  int index;
                  GLfloat rgb[4];
                  index = keyCode - XK_1;
                  rgb[0] = (GLfloat)(index & 1);
                  rgb[1] = (GLfloat)((index >> 1) & 1);
                  rgb[2] = (GLfloat)((index >> 2) & 1);
                  rgb[3] = 1.0;

                  if ((event.xkey.state & Mod1Mask) && (event.xkey.state & ShiftMask)) {
                     // Alt-Shift
                     glClearColor(0.75*rgb[0], 0.75*rgb[1], 0.75*rgb[2], 0.0);
                  }
                  else if ((event.xkey.state & ControlMask) && (event.xkey.state & ShiftMask)) {
                     // Ctrl-Shift
                     glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, rgb);
                     rgb[0] = 1.0 - rgb[0];
                     rgb[1] = 1.0 - rgb[1];
                     rgb[2] = 1.0 - rgb[2];
                     glMaterialfv(GL_BACK, GL_AMBIENT_AND_DIFFUSE, rgb);
                  }
                  else if (event.xkey.state & ShiftMask) {
                     // Shift
                     glMaterialfv(GL_BACK, GL_AMBIENT_AND_DIFFUSE, rgb);
                  }
                  else if (event.xkey.state & Mod1Mask) {
                     // Alt
                     FreePixelColour(currentTextColour);
                     currentTextColour = AllocPixelColour(0.75*rgb[0], 0.75*rgb[1], 0.75*rgb[2]);
                     XSetForeground(XStuff.display, XStuff.gc, currentTextColour);
                  }
                  else {
                     // No modifiers
                     glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, rgb);
                  }
                  Refresh();
                  break;
*/
	case '27': //XK_Escape:
                  // time to exit
//                  bQuit = True;
                  break;
            };
//            break;
 /*        case KeyRelease:
            break;
         case ButtonRelease:
            Rotator.TurnClutchOff();
            break;
         case ButtonPress:
           XQueryPointer(XStuff.display, XStuff.window, &dummyWindow1,
                        &dummyWindow2, &dummyInt1, &dummyInt2,
                        &newMouseX, &newMouseY, &StateOfModifiers);
#ifndef TRY_SOMETHING_WEIRD
            Rotator.TurnClutchOn(newMouseX, newMouseY);
#else
            Rotator.TurnClutchOn(windowWidth/2, windowHeight/2);
            Rotator.DeviceMotion(newMouseX,newMouseY);
            Refresh();
#endif
            break;
         case MotionNotify:
            if (Rotator.IsClutchOn()) {
               XQueryPointer(XStuff.display, XStuff.window, &dummyWindow1,
                        &dummyWindow2, &dummyInt1, &dummyInt2,
                        &newMouseX, &newMouseY, &StateOfModifiers);
               Rotator.DeviceMotion(newMouseX,newMouseY);
               Refresh();
            }
            break;
      }   /* switch(event */
   }   /* while (!bQuit) */
//}








void processSpecialKeys(int key, int x, int y) {

	switch(key) {

	case	GLUT_KEY_PAGE_UP:  //case XK_Page_Up:
                  sphere.IncrementNumStrips();
                  Refresh();
                  break;
	case GLUT_KEY_PAGE_DOWN:  //XK_Page_Down:
                  sphere.DecrementNumStrips();
                  Refresh();
                  break;
	case GLUT_KEY_UP:			//XK_Up:
                  frustum.Magnify(1);
                  Refresh();
                  break;
	case GLUT_KEY_DOWN:		//XK_Down:
                  frustum.Magnify(-1);
                  Refresh();
                  break;
	case GLUT_KEY_HOME:		//XK_Home:
                  frustum.Reset();
                  Refresh();
                  break;
	case GLUT_KEY_RIGHT:		//XK_Right:
                  sphere.IncrementNumStripsToDisplay();
                  Refresh();
                  break;
	case GLUT_KEY_LEFT:			//XK_Left:
                  sphere.DecrementNumStripsToDisplay();
                  Refresh();
                  break;
	case GLUT_KEY_END:			//XK_End:
                  sphere.ChangeNumHemispheresToDisplay();
                  Refresh();
                  break;
//       --        case XK_Delete:
//                  bHalfStrips = !bHalfStrips;
//                  sphere.Reconstruct();
//                  Refresh();
//                  break;
	case GLUT_KEY_F1:				//XK_F1:
                  sphere.IncrementLatitudinalResolution();
                  Refresh();
                  break;
	case GLUT_KEY_F2:				//XK_F2:
                  sphere.DecrementLatitudinalResolution();
                  Refresh();
                  break;
	case GLUT_KEY_F3:				//XK_F3:
                  sphere.IncrementLongitudinalResolution();
                  Refresh();
                  break;
	case GLUT_KEY_F4:				//XK_F4:
                  sphere.DecrementLongitudinalResolution();
                  Refresh();
                  break;
	case GLUT_KEY_F5:				//XK_F5:
                  bAnimatedEversion = !bAnimatedEversion;
                  break;
	case GLUT_KEY_F6:				//XK_F6:
                  bAnimatedRotation = !bAnimatedRotation;
                  break;
	case GLUT_KEY_F9:				//XK_F9:
                  bDisplayText = ! bDisplayText;
                  Refresh();
                  break;
	};
}
/*
#ifdef TOGGLEABLE_WINDOW_DECORATIONS
               case GLUT_KEY_F10			//XK_F10:
                  PropMotifWmHints mwm_hints;
                  Atom mwm_hints_atom;

                  bNoWindowDecorations = ! bNoWindowDecorations;
                  mwm_hints_atom  = XInternAtom(
                     XStuff.display, _XA_MOTIF_WM_HINTS, False
                  );
                  mwm_hints.flags = (int) MWM_HINTS_DECORATIONS;
                  mwm_hints.decorations = bNoWindowDecorations ? 0
                     : MWM_DECOR_ALL;
                     //: MWM_DECOR_BORDER | MWM_DECOR_TITLE | MWM_DECOR_RESIZEH
                     //| MWM_DECOR_MINIMIZE | MWM_DECOR_MAXIMIZE | MWM_DECOR_MENU;
                  XChangeProperty (
                     XStuff.display, XStuff.window, mwm_hints_atom,
                     mwm_hints_atom,
                     32, PropModeReplace, (unsigned char *) & mwm_hints,
                     PROP_MWM_HINTS_ELEMENTS
                  );
                  break;
#endif
*/







void renderScene (void) {
	   	   //byron    CreateCursorForWindow();

   // Initialize some X stuff.
   //
//byron    currentTextColour = AllocPixelColour(0.0, 0.75, 0.75);
 //byron   XSetForeground(XStuff.display, XStuff.gc, currentTextColour);

   // Initialize some OpenGL stuff.

if (bBackFaceCulling) {
// turn on back face culling
glEnable(GL_CULL_FACE);
glCullFace(bFrontFacing ? GL_BACK : GL_FRONT);
   }

   // Don't bother doing this, since the normals are
   // already normalized by us at generation time.
   // There's no need to ask OpenGL to explicitly try
   // to normalize again and again every time the
   // user rotates the view.
   //
   // glEnable(GL_NORMALIZE);   // This asks that normal vectors be normalized for us, automatically

glClearColor(0.0, 0.0, 0.0, 0.0);   // this sets the colour to use when glClear() is called
glClearDepth(1.0);
   // glDepthFunc(GL_LEQUAL);
glEnable(GL_DEPTH_TEST);

   // Don't bother doing this
   //
   // // Clear the buffers.
   // //
   // glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   // glXSwapBuffers(display, window);
   // glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   // glXSwapBuffers(display, window);

   // Setup colours & lighting properties of the materiel.
   //
GLfloat specular[] = { 1.0, 1.0, 1.0, 1.0 };
GLfloat colourFront[] = { 0.0, 1.0, 1.0, 1.0 };   // colour of front faces of object
GLfloat colourBack[] = { 1.0, 0.0, 0.0, 1.0 };    // colour of back faces of object
GLfloat shininess[] = { 50.0 };
glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, colourFront);
glMaterialfv(GL_BACK, GL_AMBIENT_AND_DIFFUSE, colourBack);
glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, shininess);

   // Setup lighting.
   //
   GLfloat lightPosition[] = { 10.0, 10.0, 10.0, 0.0 }; 
   glLightfv(GL_LIGHT0, GL_POSITION, lightPosition);
   glEnable(GL_LIGHTING);
   glEnable(GL_LIGHT0);
   GLfloat lightModelFlag[1] = { 1.0 };
   glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, lightModelFlag);


   // Initialze some of my own stuff.
   //
   Rotator.CalibrateDevice(windowWidth/2,windowHeight/2,minOfWidthAndHeight()/2);
   frustum.Init();
////   printf("RenderScene called Init\n");
   Refresh();

   // Loop for events until user quits.
   //
     };


void processMouse(int button, int state, int x, int y) {

 buttonState = button;
	
	if (state == GLUT_DOWN) 

		if (button == GLUT_LEFT_BUTTON) {
			Rotator.TurnClutchOn(windowWidth/2,windowHeight/2);
////			printf("Clutch on --processMouse\n");

			Rotator.DeviceMotion(x,y);
////			printf("DeviceMotion called --processMouse\n");

			Refresh();
		}
	
	
}

void processMouseActiveMotion(int x, int y) {

	if (buttonState == GLUT_LEFT_BUTTON) {
		Rotator.DeviceMotion(x,y);
//// printf("MouseActive -- x=%d y=%d\n",x,y);
		Refresh();
	}
}


 
// ==============================================================================

   void main(int argc, char **argv) {      //byron



   printf(
      "This program displays a sphere undergoing the Thurston eversion.\n"
      "To rotate, click and drag with the left mouse button.\n"
      "Keys:\n"
      "  keypad \"+\"/\"-\"     Increase/decrease t by delta_t\n"
      "  keypad \"*\"/\"/\"     Multiply/divide delta_t by 2\n"
      "  page up/down       Increase/decrease total number of strips\n"
      "  up/down arrows     Zoom in/out\n"

      // Pan is not yet implemented, sorry.
      // "  home               Reset zoom and pan to initial values\n"
      "  home               Reset zoom to initial value\n"

      "  right/left arrows  Increase/decrease number of strips displayed\n"
      "  end                Toggle display of lower hemisphere\n"
      "  delete             Toggle display of half-strips\n"
      "  space              Cycle through polygons/checkered polygons/bands/points\n"
      "  F1/F2              Increase/decrease number of latitudinal patches\n"
      "  F3/F4              Increase/decrease number of longitudinal patches\n"
      "  F5/F6              Toggle animated eversion/rotation\n"
      "  F9                 Toggle text\n"
#ifdef TOGGLEABLE_WINDOW_DECORATIONS
      "  F10                Toggle window decorations\n"
      "                     (requires Motif-like window manager)\n"
#endif
      "  \"s\"                Toggle smooth/flat shading\n"
      "  \"b\"                Toggle backface culling\n"
      "  \"f\"                Toggle which faces are front facing\n"
      "  \"r\"                Cycle through rotation modes\n"
      "  \"1\"-\"8\"            Select colour of ...\n"
      "                       front faces (no modifiers), back faces (use Shift),\n"
      "                       front and back faces (use Ctrl-Shift),\n"
      "                       text (use Alt), background (use Alt-Shift)\n"
      "  escape             Quit\n"
      "Press Enter to begin ... "
   );
   getchar();


    glutInit(&argc, argv);												//byron
	glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGBA);			//byron
	glutInitWindowPosition(100,100);									//byron
	glutInitWindowSize(600,600);										//byron
	glutCreateWindow("Sphere Eversion");								//byron
	glutDisplayFunc(renderScene);
	glutKeyboardFunc(processNormalKeys);
	glutSpecialFunc(processSpecialKeys);
	glutReshapeFunc(changeSize);

//mouse
	glutMouseFunc(processMouse);
	glutMotionFunc(processMouseActiveMotion);
//	glutPassiveMotionFunc(processMousePassiveMotion);
//	glutEntryFunc(processMouseEntry);


    glutMainLoop();
















   // Clean up.
   //
//byron    FreePixelColour(currentTextColour);
//byron    if (NULL != XStuff.bigFont)
//byron       XUnloadFont(XStuff.display, XStuff.bigFont->fid);
//byron    if (NULL != XStuff.smallFont)
//byron       XUnloadFont(XStuff.display, XStuff.smallFont->fid);
//byron    XFreeGC(XStuff.display, XStuff.gc);
//byron    XFreeCursor(XStuff.display, XStuff.cursor);
//byron    XCloseDisplay(XStuff.display);
}

